package cdc.issues.io;

import java.io.File;
import java.io.OutputStream;

import cdc.util.lang.Checks;
import cdc.util.lang.FailureReaction;
import cdc.util.lang.Introspection;

/**
 * Class that can be used to create instances of:
 * <ul>
 * <li>{@link IssuesWriter}
 * <li>{@link IssuesReader}
 * </ul>
 *
 * @author Damien Carbonne
 */
public class IssuesIoFactory {
    private final IssuesIoFactoryFeatures features;

    public IssuesIoFactory(IssuesIoFactoryFeatures features) {
        this.features = features;
    }

    public IssuesIoFactoryFeatures getFeatures() {
        return features;
    }

    public boolean canExportTo(File file) {
        if (file == null) {
            return false;
        } else {
            return IssuesFormat.from(file) != null;
        }
    }

    public IssuesStreamWriter createIssuesStreamWriter(File file,
                                                       OutSettings settings) {
        Checks.isNotNull(file, "file");

        final IssuesFormat format = IssuesFormat.from(file);
        if (format != null) {
            switch (format) {
            case CSV:
            case ODS:
            case XLS:
            case XLSM:
            case XLSX:
                return createIssuesStreamWriter("cdc.issues.core.io.WorkbookIssuesStreamWriter", file, settings);
            case JSON:
                return createIssuesStreamWriter("cdc.issues.core.io.JsonIssuesStreamWriter", file, settings);
            case XML:
                return createIssuesStreamWriter("cdc.issues.core.io.XmlIssuesStreamWriter", file, settings);
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Can not create an IssuesStreamWriter for " + file);
    }

    public IssuesStreamWriter createIssuesStreamWriter(OutputStream out,
                                                       IssuesFormat format,
                                                       OutSettings settings) {
        if (format != null) {
            switch (format) {
            case CSV:
            case ODS:
            case XLS:
            case XLSM:
            case XLSX:
                return createIssuesStreamWriter("cdc.issues.core.io.WorkbookIssuesStreamWriter", out, format, settings);
            case JSON:
                return createIssuesStreamWriter("cdc.issues.core.io.JsonIssuesStreamWriter", out, format, settings);
            case XML:
                return createIssuesStreamWriter("cdc.issues.core.io.XmlIssuesStreamWriter", out, format, settings);
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Can not create an IssuesStreamWriter for " + format);
    }

    /**
     * Creates an IssuesWriter.
     * <p>
     * See {@link IssuesFormat} for a list of supported formats.
     *
     * @param file The file for which an IssuesWriter must be created.
     * @return A new IssueWriter which can produce {@code file}.
     * @throws IllegalArgumentException When {@code file} is {@code null} or has an unrecognized extension.
     */
    public IssuesWriter createIssuesWriter(File file) {
        Checks.isNotNull(file, "file");

        final IssuesFormat format = IssuesFormat.from(file);
        if (format != null) {
            switch (format) {
            case CSV:
            case ODS:
            case XLS:
            case XLSM:
            case XLSX:
            case JSON:
            case XML:
                return createIssuesWriter("cdc.issues.core.io.IssuesWriterImpl", file);
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Can not create an IssuesWriter for " + file);
    }

    public IssuesWriter createIssuesWriter(OutputStream out,
                                           IssuesFormat format) {
        if (format != null) {
            switch (format) {
            case CSV:
            case ODS:
            case XLS:
            case XLSM:
            case XLSX:
            case JSON:
            case XML:
                return createIssuesWriter("cdc.issues.core.io.IssuesWriterImpl", out, format);
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Can not create an IssuesWriter for " + format);
    }

    private IssuesStreamWriter createIssuesStreamWriter(String className,
                                                        File file,
                                                        OutSettings settings) {
        final Class<? extends IssuesStreamWriter> cls =
                Introspection.uncheckedCast(Introspection.getClass(className,
                                                                   IssuesStreamWriter.class,
                                                                   FailureReaction.FAIL));
        final Class<?>[] parameterTypes = { File.class, OutSettings.class, IssuesIoFactoryFeatures.class };
        return Introspection.newInstance(cls,
                                         parameterTypes,
                                         FailureReaction.FAIL,
                                         file,
                                         settings,
                                         getFeatures());
    }

    private IssuesStreamWriter createIssuesStreamWriter(String className,
                                                        OutputStream out,
                                                        IssuesFormat format,
                                                        OutSettings settings) {
        final Class<? extends IssuesStreamWriter> cls =
                Introspection.uncheckedCast(Introspection.getClass(className,
                                                                   IssuesStreamWriter.class,
                                                                   FailureReaction.FAIL));
        final Class<?>[] parameterTypes =
                { OutputStream.class, IssuesFormat.class, OutSettings.class, IssuesIoFactoryFeatures.class };
        return Introspection.newInstance(cls, parameterTypes, FailureReaction.FAIL, out, format, settings, getFeatures());
    }

    private IssuesWriter createIssuesWriter(String className,
                                            File file) {
        final Class<? extends IssuesWriter> cls =
                Introspection.uncheckedCast(Introspection.getClass(className,
                                                                   IssuesWriter.class,
                                                                   FailureReaction.FAIL));
        final Class<?>[] parameterTypes = { File.class, IssuesIoFactoryFeatures.class };
        return Introspection.newInstance(cls,
                                         parameterTypes,
                                         FailureReaction.FAIL,
                                         file,
                                         getFeatures());
    }

    private IssuesWriter createIssuesWriter(String className,
                                            OutputStream out,
                                            IssuesFormat format) {
        final Class<? extends IssuesWriter> cls =
                Introspection.uncheckedCast(Introspection.getClass(className,
                                                                   IssuesWriter.class,
                                                                   FailureReaction.FAIL));
        final Class<?>[] parameterTypes = { OutputStream.class, IssuesFormat.class, IssuesIoFactoryFeatures.class };
        return Introspection.newInstance(cls, parameterTypes, FailureReaction.FAIL, out, format, getFeatures());
    }

    public boolean canImportFrom(File file) {
        if (file == null) {
            return false;
        } else {
            return IssuesFormat.from(file) != null;
        }
    }

    public IssuesReader createIssuesReader(File file) {
        Checks.isNotNull(file, "file");

        final IssuesFormat format = IssuesFormat.from(file);
        if (format != null) {
            switch (format) {
            case CSV:
            case ODS:
            case XLS:
            case XLSM:
            case XLSX:
                return createIssuesReader("cdc.issues.core.io.WorkbookIssuesReader", file);
            case JSON:
                return createIssuesReader("cdc.issues.core.io.JsonIssuesReader", file);
            case XML:
                return createIssuesReader("cdc.issues.core.io.XmlIssuesReader", file);
            default:
                break;
            }
        }
        throw new IllegalArgumentException("Can not create an IssuesReader for " + file);
    }

    private IssuesReader createIssuesReader(String className,
                                            File file) {
        final Class<? extends IssuesReader> cls =
                Introspection.uncheckedCast(Introspection.getClass(className,
                                                                   IssuesReader.class,
                                                                   FailureReaction.FAIL));
        final Class<?>[] parameterTypes = { File.class, IssuesIoFactoryFeatures.class };
        return Introspection.newInstance(cls, parameterTypes, FailureReaction.FAIL, file, getFeatures());
    }
}
