package cdc.issues.rules;

import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Optional;
import java.util.Set;

import cdc.issues.rules.IssuesDetector.Factory;
import cdc.util.lang.Checks;

/**
 * Implementation of {@link RulesCatalog} that aggregates delegates catalogs.
 * <p>
 * This implementation does not cache any data.
 *
 * @author Damien Carbonne
 */
public class CompositeRulesCatalog implements RulesCatalog {
    private final Set<RulesCatalog> catalogs = new LinkedHashSet<>();

    /**
     * Adds a new delegate catalog.
     *
     * @param catalog The catalog to add.
     * @return This catalog.
     */
    public CompositeRulesCatalog add(RulesCatalog catalog) {
        Checks.isNotNull(catalog, "catalog");

        catalogs.add(catalog);

        return this;
    }

    @Override
    public Set<String> getDomains() {
        final Set<String> set = new HashSet<>();
        for (final RulesCatalog catalog : catalogs) {
            set.addAll(catalog.getDomains());
        }
        return set;
    }

    @Override
    public Set<Rule> getRules() {
        final Set<Rule> set = new HashSet<>();
        for (final RulesCatalog catalog : catalogs) {
            set.addAll(catalog.getRules());
        }
        return set;
    }

    @Override
    public Optional<Rule> getRule(RuleId id) {
        for (final RulesCatalog catalog : catalogs) {
            final Optional<Rule> rule = catalog.getRule(id);
            if (rule.isPresent()) {
                return rule;
            }
        }
        return Optional.empty();
    }

    @Override
    public Set<Factory<?>> getFactories() {
        final Set<Factory<?>> set = new HashSet<>();
        for (final RulesCatalog catalog : catalogs) {
            set.addAll(catalog.getFactories());
        }
        return set;
    }

    @Override
    public <T> Set<Factory<T>> getFactories(Class<T> dataClass) {
        final Set<Factory<T>> set = new HashSet<>();
        for (final RulesCatalog catalog : catalogs) {
            set.addAll(catalog.getFactories(dataClass));
        }
        return set;
    }

    @Override
    public <T> Optional<Factory<T>> getFactory(Rule rule,
                                               Class<T> dataClass) {
        for (final RulesCatalog catalog : catalogs) {
            final Optional<Factory<T>> factory = catalog.getFactory(rule, dataClass);
            if (factory.isPresent()) {
                return factory;
            }
        }
        return Optional.empty();
    }
}