package cdc.issues.api;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import cdc.util.lang.Checks;

/**
 * Base class used to describe an Issue.
 * <p>
 * <b>Node:</b> This class can be specialized if necessary.
 * <p>
 * An issue may have several targets.<br>
 * For example, when there is a compliance issue between several things,
 * one can not tell which one is at fault.
 *
 * @author Damien Carbonne
 */
public class Issue {
    private final Instant timestamp;
    private final IssueId id;
    private final IssueSeverity severity;
    private final String description;
    private final Throwable cause;

    /**
     * Creates an Issue.
     *
     * @param timestamp The optional timestamp.
     * @param name The issue name.
     * @param params The parameters.
     * @param locations The target locations.
     * @param severity The issue severity.
     * @param description The issue description.
     * @param cause The optional cause.
     */
    protected Issue(Instant timestamp,
                    String domain,
                    String name,
                    IssueParams params,
                    List<? extends IssueLocation> locations,
                    IssueSeverity severity,
                    String description,
                    Throwable cause) {
        this.timestamp = timestamp == null
                ? Instant.now()
                : timestamp;
        this.id = new IssueId(domain,
                              name,
                              params,
                              locations.toArray(new IssueLocation[locations.size()]));
        this.severity = Checks.isNotNull(severity, "severity");
        this.description = Checks.isNotNull(description, "description");
        this.cause = cause;
    }

    protected Issue(String domain,
                    String name,
                    IssueParams params,
                    List<? extends IssueLocation> locations,
                    IssueSeverity severity,
                    String description,
                    Throwable cause) {
        this(null,
             domain,
             name,
             params,
             locations,
             severity,
             description,
             cause);
    }

    protected Issue(Instant timestamp,
                    String domain,
                    Enum<?> name,
                    IssueParams params,
                    List<? extends IssueLocation> locations,
                    IssueSeverity severity,
                    String description,
                    Throwable cause) {
        this(timestamp,
             domain,
             name.name(),
             params,
             locations,
             severity,
             description,
             cause);
    }

    protected Issue(String domain,
                    Enum<?> name,
                    IssueParams params,
                    List<? extends IssueLocation> locations,
                    IssueSeverity severity,
                    String description,
                    Throwable cause) {
        this(null,
             domain,
             name,
             params,
             locations,
             severity,
             description,
             cause);
    }

    /**
     * @return The Instant at which this issue was created.
     */
    public final Instant getTimestamp() {
        return timestamp;
    }

    /**
     * @return The id of this issue.
     */
    public IssueId getId() {
        return id;
    }

    /**
     * @return The RuleId of this issue.
     */
    public RuleId getRuleId() {
        return id.getRuleId();
    }

    /**
     * @return The domain of this issue.
     */
    public String getDomain() {
        return id.getDomain();
    }

    /**
     * @return The name of this issue.
     */
    public String getName() {
        return id.getName();
    }

    public <T extends Enum<T>> T getName(Class<T> typeClass) {
        return id.getName(typeClass);
    }

    public IssueParams getParams() {
        return id.getParams();
    }

    /**
     * @return The severity of this issue.
     */
    public final IssueSeverity getSeverity() {
        return severity;
    }

    /**
     * @return The description of this issue.
     */
    public final String getDescription() {
        return description;
    }

    /**
     * @return The target locations of this issue.
     */
    public IssueLocation[] getLocations() {
        return id.getLocations();
    }

    /**
     * @return The number of target locations of this issue.
     */
    public final int getNumberOfLocations() {
        return id.getLocations().length;
    }

    public IssueLocation getLocationAt(int index) {
        return id.getLocations()[index];
    }

    public <L extends IssueLocation> L getLocationAt(int index,
                                                     Class<L> cls) {
        return cls.cast(getLocationAt(index));
    }

    /**
     * @return The cause of this issue, or {@code null}.
     */
    public Throwable getCause() {
        return cause;
    }

    @Override
    public int hashCode() {
        return Objects.hash(timestamp,
                            id,
                            severity,
                            description,
                            cause);
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (!(object instanceof Issue)) {
            return false;
        }
        final Issue other = (Issue) object;
        return Objects.equals(this.timestamp, other.timestamp)
                && Objects.equals(this.id, other.id)
                && this.severity == other.severity
                && Objects.equals(this.description, other.description)
                && Objects.equals(this.cause, other.cause);
    }

    @Override
    public String toString() {
        return getTimestamp()
                + " " + getName()
                + " " + getSeverity()
                + " " + getDescription()
                + (getCause() == null ? "" : " " + getCause())
                + " " + Arrays.toString(getLocations());
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        protected Instant timestamp = null;
        protected String domain;
        protected String name;
        protected IssueParams params = IssueParams.NO_PARAMS;
        protected IssueSeverity severity;
        protected String description;
        protected final List<IssueLocation> locations = new ArrayList<>();
        protected Throwable cause = null;

        protected Builder() {
        }

        protected Builder self() {
            return this;
        }

        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return self();
        }

        public Builder ruleId(RuleId ruleId) {
            this.domain = ruleId.getDomain();
            this.name = ruleId.getName();
            return self();
        }

        public Builder domain(String domain) {
            this.domain = domain;
            return self();
        }

        public Builder name(String name) {
            this.name = name;
            return self();
        }

        public Builder name(Enum<?> name) {
            this.name = name.name();
            return self();
        }

        public Builder params(IssueParams params) {
            this.params = params;
            return self();
        }

        public Builder addLocation(IssueLocation location) {
            this.locations.add(location);
            return self();
        }

        public Builder locations(IssueLocation[] locations) {
            this.locations.clear();
            Collections.addAll(this.locations, locations);
            return this;
        }

        public Builder locations(List<IssueLocation> locations) {
            this.locations.clear();
            this.locations.addAll(locations);
            return this;
        }

        public Builder severity(IssueSeverity severity) {
            this.severity = severity;
            return self();
        }

        public Builder description(String description) {
            this.description = description;
            return self();
        }

        public Builder cause(Throwable cause) {
            this.cause = cause;
            return self();
        }

        public Issue build() {
            return new Issue(timestamp,
                             domain,
                             name,
                             params,
                             locations,
                             severity,
                             description,
                             cause);
        }
    }

    public abstract static class AbstractIssueBuilder<B extends AbstractIssueBuilder<B, I>, I extends Issue> extends Builder {
        protected AbstractIssueBuilder() {
        }

        @Override
        protected abstract B self();

        @Override
        public Builder timestamp(Instant timestamp) {
            super.timestamp(timestamp);
            return self();
        }

        @Override
        public B ruleId(RuleId ruleId) {
            super.ruleId(ruleId);
            return self();
        }

        @Override
        public B domain(String domain) {
            super.domain(domain);
            return self();
        }

        @Override
        public B name(String name) {
            super.name(name);
            return self();
        }

        @Override
        public B name(Enum<?> name) {
            super.name(name);
            return self();
        }

        @Override
        public B params(IssueParams params) {
            super.params(params);
            return self();
        }

        @Override
        public B addLocation(IssueLocation location) {
            super.addLocation(location);
            return self();
        }

        @Override
        public B locations(IssueLocation[] locations) {
            super.locations(locations);
            return self();
        }

        @Override
        public B locations(List<IssueLocation> locations) {
            super.locations(locations);
            return self();
        }

        @Override
        public B severity(IssueSeverity severity) {
            super.severity(severity);
            return self();
        }

        @Override
        public B description(String description) {
            super.description(description);
            return self();
        }

        @Override
        public B cause(Throwable cause) {
            super.cause(cause);
            return self();
        }

        @Override
        public abstract I build();
    }
}