package cdc.issues.api.rules;

import java.util.Set;

import cdc.issues.api.FormalParams;
import cdc.issues.api.Issue;
import cdc.issues.api.IssuesHandler;
import cdc.issues.api.Params;
import cdc.issues.api.locations.Location;

/**
 * Interface implemented by classes that can detect issues.
 *
 * @author Damien Carbonne
 *
 * @param <T> The data type onto which issue can be detected.
 */
public interface IssuesDetector<T> {

    /**
     * Interface implemented by classes that can describe and create an IssueDetector.
     *
     * @author Damien Carbonne
     *
     * @param <T> The data type onto which issue can be detected.
     */
    public interface Descriptor<T> {
        /**
         * @return The class of data that can be analyzed.
         */
        public Class<T> getDataClass();

        /**
         * @return The set of all supported Rules.
         */
        public Set<Rule> getRules();

        /**
         * @return The description of parameters that can or must be
         *         used to create an associated IssuesDetector.
         */
        public FormalParams getFormalParams();

        /**
         * Creates and configures an IssuesDetector to analyze some rules.
         *
         * @param params The effective parameters to use.
         *            Meaningless parameters can be passed. They musty be ignored.
         * @param rules The rules for which analysis must be performed.<br>
         *            It must be a subset of supported rules.
         * @return A newly created IssuesDetector, configured with {@code params} and {@code rules}.
         */
        public IssuesDetector<T> create(Params params,
                                        Set<Rule> rules);
    }

    /**
     * @return The descriptor that was used to create this IssuesDetector.
     */
    public Descriptor<T> getDescriptor();

    /**
     * @return The class of data that can be analyzed.
     */
    public default Class<T> getDataClass() {
        return getDescriptor().getDataClass();
    }

    /**
     * @return The set of enabled Rules.
     */
    public Set<Rule> getEnabledRules();

    /**
     * @return The parameters used to configure this detector.
     */
    public Params getParams();

    /**
     * Analyzes one data.
     *
     * @param data The data to analyze.
     * @param location The data location.
     * @param issuesHandler The issues handler that must be used to store issues.
     */
    public void analyze(T data,
                        Location location,
                        IssuesHandler<Issue> issuesHandler);

    public static String toString(IssuesDetector<?> detector) {
        final StringBuilder builder = new StringBuilder();
        builder.append("IssuesDetector<")
               .append(detector.getDataClass().getSimpleName())
               .append(">(");
        boolean first = true;
        for (final Rule rule : detector.getEnabledRules()) {
            if (first) {
                first = false;
            } else {
                builder.append(",");
            }
            builder.append(rule.getName());
        }
        builder.append(")");
        return builder.toString();
    }
}