package cdc.issues.api.rules;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.Spliterator;
import java.util.function.Consumer;

import cdc.issues.api.Issue;
import cdc.issues.api.IssuesHandler;
import cdc.issues.api.Params;
import cdc.issues.api.locations.LocatedData;
import cdc.util.events.ProgressController;
import cdc.util.events.ProgressSupplier;
import cdc.util.lang.Checks;
import cdc.util.lang.Introspection;
import cdc.util.lang.NotFoundException;
import cdc.util.tuples.Tuple2;

/**
 * Catalog of {@link Rule}s and {@link IssuesDetector.Descriptor}s.
 *
 * @author Damien Carbonne
 */
public class RulesCatalog {
    /**
     * (RuleId -> Rule) map.
     */
    private final Map<RuleId, Rule> ruleIdToRule = new HashMap<>();

    /**
     * Set of rule domains.
     */
    private final Map<String, Set<Rule>> domainToRules = new HashMap<>();

    /**
     * Set of descriptors.
     */
    private final Set<IssuesDetector.Descriptor<?>> descriptors = new HashSet<>();

    /**
     * ((RuleId, Data class) -> Descriptor) map.
     */
    private final Map<Tuple2<RuleId, Class<?>>, IssuesDetector.Descriptor<?>> ruleIdDataClassToDescriptor = new HashMap<>();

    /**
     * (dataClass -> Set<Descriptor>) map.
     */
    private final Map<Class<?>, Set<IssuesDetector.Descriptor<?>>> dataClassToDescriptors = new HashMap<>();

    public RulesCatalog() {
        super();
    }

    public void register(Rule rule) {
        Checks.isNotNull(rule, "rule");
        Checks.doesNotContainKey(ruleIdToRule, rule.getId(), "rules");

        ruleIdToRule.put(rule.getId(), rule);
        final Set<Rule> set = domainToRules.computeIfAbsent(rule.getId().getDomain(), k -> new HashSet<>());
        set.add(rule);
    }

    public <T> void register(IssuesDetector.Descriptor<T> descriptor) {
        Checks.isNotNull(descriptor, "descriptor");
        Checks.doesNotContain(descriptors, descriptor, "descriptors");

        descriptors.add(descriptor);
        final Set<IssuesDetector.Descriptor<?>> set =
                dataClassToDescriptors.computeIfAbsent(descriptor.getDataClass(), k -> new HashSet<>());
        set.add(descriptor);

        for (final Rule rule : descriptor.getRules()) {
            final Tuple2<RuleId, Class<?>> key = new Tuple2<>(rule.getId(), descriptor.getDataClass());
            if (!isRegistered(rule)) {
                register(rule);
            }
            if (ruleIdDataClassToDescriptor.containsKey(key)) {
                throw new IllegalArgumentException("Duplicate descriptor for " + key);
            } else {
                ruleIdDataClassToDescriptor.put(key, descriptor);
            }
        }
    }

    /**
     * @return A set of domains for which rules are registered.
     */
    public Set<String> getDomains() {
        return domainToRules.keySet();
    }

    public boolean isRegistered(Rule rule) {
        return ruleIdToRule.containsKey(rule.getId());
    }

    /**
     * @return A set of known RuleIds.
     */
    public Set<RuleId> getRuleIds() {
        return ruleIdToRule.keySet();
    }

    /**
     * @return A set of known Rules.
     */
    public Collection<Rule> getRules() {
        return ruleIdToRule.values();
    }

    /**
     * Returns a set rules associated to a domain.
     *
     * @param domain The domain.
     * @return A set of rules associated to a domain and registered in this catalog.
     */
    public Set<Rule> getRules(String domain) {
        return domainToRules.getOrDefault(domain, Collections.emptySet());
    }

    /**
     * Returns the rule associated to an id or {@code null}.
     *
     * @param id The RuleId.
     * @return The Rule associated to {@code id} or {@code null}.
     */
    public Rule getRuleOrNull(RuleId id) {
        return ruleIdToRule.get(id);
    }

    /**
     * @return A set of all registered Descriptors.
     */
    public Set<IssuesDetector.Descriptor<?>> getDescriptpors() {
        return descriptors;
    }

    /**
     * Returns a set of all Descriptors registered for a data class.
     *
     * @param <T> The data type.
     * @param dataClass The data class.
     * @return A set of all Descriptors registered for {@code dataClass}.
     */
    public <T> Set<IssuesDetector.Descriptor<T>> getDescriptors(Class<T> dataClass) {
        final Set<IssuesDetector.Descriptor<?>> set =
                dataClassToDescriptors.getOrDefault(dataClass, Collections.emptySet());
        return Introspection.uncheckedCast(set);
    }

    /**
     * Returns {@code true} if an {@link IssuesDetector.Descriptor}
     * is registered for a Rule and a Class.
     *
     * @param rule The rule.
     * @param dataClass The data class
     * @return {@code true} if an {@link IssuesDetector.Descriptor} is registered
     *         for {@code rule} and {@code dataClass}.
     */
    public boolean hasDescriptor(Rule rule,
                                 Class<?> dataClass) {
        final Tuple2<RuleId, Class<?>> key = new Tuple2<>(rule.getId(), dataClass);
        return ruleIdDataClassToDescriptor.containsKey(key);
    }

    /**
     * Returns the {@link IssuesDetector.Descriptor} associated to a Rule and
     * a Class, or {@code null}.
     *
     * @param <T> The data type.
     * @param rule The rule.
     * @param dataClass The data class.
     * @return The {@link IssuesDetector.Descriptor} associated to {@code rule}
     *         and {@code dataClass}, or {@code null}.
     */
    public <T> IssuesDetector.Descriptor<T> getDescriptorOrNull(Rule rule,
                                                                Class<T> dataClass) {
        final Tuple2<RuleId, Class<?>> key = new Tuple2<>(rule.getId(), dataClass);
        final IssuesDetector.Descriptor<T> descriptor =
                Introspection.uncheckedCast(ruleIdDataClassToDescriptor.get(key));
        return descriptor;
    }

    /**
     * Returns the {@link IssuesDetector.Descriptor} associated to a Rule and a Class.
     *
     * @param <T> The data type.
     * @param rule The rule.
     * @param dataClass The data class.
     * @return The {@link IssuesDetector.Descriptor} associated to {@code rule} and {@code dataClass}.
     * @throws NotFoundException When no matching Descriptor could be found.
     */
    public <T> IssuesDetector.Descriptor<T> getDescriptor(Rule rule,
                                                          Class<T> dataClass) {
        final IssuesDetector.Descriptor<T> descriptor = getDescriptorOrNull(rule, dataClass);
        if (descriptor == null) {
            throw new NotFoundException("No descriptor found for " + rule.getName() + " " + dataClass);
        } else {
            return descriptor;
        }
    }

    /**
     * Creates and configures one IssuesDetector for one rule and parameters.
     *
     * @param <T> The data type.
     * @param rule The rule.
     * @param dataClass The data class.
     * @param params The detector effective parameters.
     * @return An IssueDetector matching {@code rule} and {@code dataClass} and configured with {@code params}.
     * @throws NotFoundException When no matching IssuesDetector could be found.
     */
    public <T> IssuesDetector<T> createIssuesDetector(Rule rule,
                                                      Class<T> dataClass,
                                                      Params params) {
        final IssuesDetector.Descriptor<T> descriptor = getDescriptor(rule, dataClass);
        final Set<Rule> rules = new HashSet<>();
        rules.add(rule);
        return descriptor.create(params, rules);
    }

    /**
     * Creates and configures an appropriate IssuesDetector, and applies it to data provided by a Spliterator.
     *
     * @param <T> The data type.
     * @param rule The rule.
     * @param dataClass The data class.
     * @param params The rule effective parameters.
     * @param spliterator The data spliterator.
     * @param issuesHandler The issues handler that will collect issues.
     * @param controller The progress controller.
     */
    public <T> void apply(Rule rule,
                          Class<T> dataClass,
                          Params params,
                          Spliterator<LocatedData<T>> spliterator,
                          IssuesHandler<Issue> issuesHandler,
                          ProgressController controller) {
        final IssuesDetector<T> detector = createIssuesDetector(rule, dataClass, params);

        apply(detector, spliterator, issuesHandler, controller);
    }

    /**
     * Create all possible IssueDectectors for a data class, configures them with all supported rules,
     * and apply them to data provided by a Spliterator.
     * <p>
     * <b>Note:</b> The passed params must be usable for all created detectors,
     * each using the appropriate parameters and ignoring those that are meaningless.
     *
     * @param <T> The data type.
     * @param dataClass The data class.
     * @param params The rule effective parameters.
     * @param spliterator The data spliterator.
     * @param issuesHandler The issues handler that will collect issues.
     * @param controller The progress controller.
     */
    public <T> void apply(Class<T> dataClass,
                          Params params,
                          Spliterator<LocatedData<T>> spliterator,
                          IssuesHandler<Issue> issuesHandler,
                          ProgressController controller) {
        for (final IssuesDetector.Descriptor<T> descriptor : getDescriptors(dataClass)) {
            final IssuesDetector<T> detector = descriptor.create(params, descriptor.getRules());
            apply(detector, spliterator, issuesHandler, controller);
        }
    }

    /**
     * Applies an IssuesDetector to data provided by a Spliterator.
     *
     * @param <T> The data type.
     * @param detector The detector.
     * @param spliterator The data spliterator.
     * @param issuesHandler The issues handler that will collect issues.
     * @param controller The progress controller.
     */
    public static <T> void apply(IssuesDetector<T> detector,
                                 Spliterator<LocatedData<T>> spliterator,
                                 IssuesHandler<Issue> issuesHandler,
                                 ProgressController controller) {
        Checks.isNotNull(detector, "detector");
        Checks.isNotNull(spliterator, "spliterator");
        Checks.isNotNull(issuesHandler, "issuesHandler");
        Checks.isNotNull(controller, "controller");

        final ProgressSupplier progress = new ProgressSupplier(controller);
        progress.reset(spliterator.estimateSize(), IssuesDetector.toString(detector));

        final Consumer<LocatedData<T>> consumer = ld -> {
            detector.analyze(ld.getData(), ld.getLocation(), issuesHandler);
        };

        boolean next = true;
        while (next) {
            next = next && spliterator.tryAdvance(consumer);
            next = next && !controller.isCancelled();
            progress.incrementValue();
        }
    }
}