package cdc.issues.rules;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import cdc.issues.IssueSeverity;
import cdc.issues.Params;
import cdc.util.lang.Checks;

public interface ProfileConfig {
    /**
     * @return The set of rule ids.
     */
    public Set<RuleId> getRuleIds();

    /**
     * @param ruleId The rule id.
     * @return The config of rule identified by {@code ruleId}.
     */
    public Optional<RuleConfig> getRuleConfig(RuleId ruleId);

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private static final String RULE_ID = "ruleId";
        private static final String RULE_CONFIG = "ruleConfig";

        final Map<RuleId, RuleConfig> configs = new HashMap<>();

        Builder() {
        }

        protected Builder self() {
            return this;
        }

        public Builder addIfMissing(RuleId ruleId) {
            if (!configs.containsKey(ruleId)) {
                set(ruleId, RuleConfig.DEFAULT);
            }
            return self();
        }

        public Builder set(RuleId ruleId,
                           RuleConfig ruleConfig) {
            Checks.isNotNull(ruleId, RULE_ID);
            Checks.isNotNull(ruleConfig, RULE_CONFIG);

            configs.put(ruleId, ruleConfig);

            return self();
        }

        public Builder setEnabled(RuleId ruleId,
                                  boolean enabled) {
            addIfMissing(ruleId);
            configs.put(ruleId,
                        RuleConfig.builder()
                                  .set(configs.get(ruleId))
                                  .enabled(enabled)
                                  .build());
            return self();
        }

        public Builder setCustomizedSeverity(RuleId ruleId,
                                             IssueSeverity customizedSeverity) {
            addIfMissing(ruleId);
            configs.put(ruleId,
                        RuleConfig.builder()
                                  .set(configs.get(ruleId))
                                  .customizedSeverity(customizedSeverity)
                                  .build());
            return self();
        }

        public Builder setParams(RuleId ruleId,
                                 Params params) {
            addIfMissing(ruleId);
            configs.put(ruleId,
                        RuleConfig.builder()
                                  .set(configs.get(ruleId))
                                  .params(params)
                                  .build());
            return self();
        }

        public ProfileConfig build() {
            return new ProfileConfigImpl(this);
        }
    }
}

class ProfileConfigImpl implements ProfileConfig {
    private final Map<RuleId, RuleConfig> configs;

    ProfileConfigImpl(ProfileConfig.Builder builder) {
        this.configs = Map.copyOf(builder.configs);
    }

    @Override
    public Set<RuleId> getRuleIds() {
        return configs.keySet();
    }

    @Override
    public Optional<RuleConfig> getRuleConfig(RuleId ruleId) {
        return Optional.ofNullable(configs.get(ruleId));
    }
}