package cdc.issues;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import cdc.issues.answers.IssueAnswer;
import cdc.issues.rules.Profile;

/**
 * Interface describing a Project.
 *
 * @author Damien Carbonne
 */
public interface Project extends MetasItem, LabelsItem {
    /**
     * @return The name of this project.
     */
    public String getName();

    /**
     * @return The description of this project.
     */
    public String getDescription();

    /**
     * @return The profile of this project.
     */
    public Optional<Profile> getProfile();

    /**
     * @return The data sources contained in this project.
     */
    @Deprecated(since = "2025-03-29", forRemoval = true)
    public Iterable<DataSource<?>> getDataSources();

    /**
     * @return The associated snapshots.
     */
    public List<? extends Snapshot> getSnapshots();

    /**
     * @return The associated answers.
     */
    public Set<? extends IssueAnswer> getAnswers();

    /**
     * @param id The issue id.
     * @return The answer having an {@code id} or {@code null}.
     */
    public Optional<IssueAnswer> getAnswer(IssueId id);

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder implements MetasBuilding<Builder>, LabelsBuilding<Builder> {
        String name;
        String description;
        final Metas.Builder metas = Metas.builder();
        Labels labels = Labels.NO_LABELS;
        Profile profile;
        final Map<IssueId, IssueAnswer> idToAnswers = new HashMap<>();
        final Set<IssueAnswer> answers = new HashSet<>();

        Builder() {
        }

        public Builder name(String name) {
            this.name = name;
            return self();
        }

        public Builder description(String description) {
            this.description = description;
            return self();
        }

        @Override
        public Builder meta(String name,
                            String value) {
            this.metas.meta(name, value);
            return self();
        }

        @Override
        public Builder meta(String name,
                            String value,
                            String separator) {
            this.metas.meta(name, value, separator);
            return self();
        }

        @Override
        public Builder metas(Metas metas) {
            this.metas.metas(metas);
            return self();
        }

        @Override
        public Builder labels(Labels labels) {
            this.labels = labels;
            return self();
        }

        public Builder profile(Profile profile) {
            this.profile = profile;
            return self();
        }

        public Builder answer(IssueAnswer answer) {
            final IssueAnswer current = idToAnswers.get(answer.getIssueId());
            answers.remove(current);
            answers.add(answer);
            idToAnswers.put(answer.getIssueId(), answer);
            return self();
        }

        public Builder answers(Collection<? extends IssueAnswer> answers) {
            for (final IssueAnswer answer : answers) {
                answer(answer);
            }
            return self();
        }

        public Project build() {
            return new ProjectImpl(this);
        }
    }
}

class ProjectImpl implements Project {
    private final String name;
    private final String description;
    private final Metas metas;
    private final Labels labels;
    private final Optional<Profile> profile;
    final List<Snapshot> snapshots = new ArrayList<Snapshot>();
    private final Map<IssueId, IssueAnswer> idToAnswers;
    private final Set<IssueAnswer> answers;

    ProjectImpl(Builder builder) {
        this.name = builder.name;
        this.description = builder.description;
        this.metas = builder.metas.build();
        this.labels = builder.labels;
        this.profile = Optional.ofNullable(builder.profile);
        this.idToAnswers = Map.copyOf(builder.idToAnswers);
        this.answers = Set.copyOf(builder.answers);
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getDescription() {
        return description;
    }

    @Override
    public Metas getMetas() {
        return metas;
    }

    @Override
    public Labels getLabels() {
        return labels;
    }

    @Override
    public Optional<Profile> getProfile() {
        return profile;
    }

    @Override
    public Iterable<DataSource<?>> getDataSources() {
        return Collections.emptyList();
    }

    @Override
    public List<? extends Snapshot> getSnapshots() {
        return snapshots;
    }

    @Override
    public Set<? extends IssueAnswer> getAnswers() {
        return answers;
    }

    @Override
    public Optional<IssueAnswer> getAnswer(IssueId id) {
        return Optional.ofNullable(idToAnswers.get(id));
    }
}