package cdc.issues.answers;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import cdc.issues.Issue;
import cdc.issues.IssueId;
import cdc.util.lang.Checks;

/**
 * Interface giving access to issues and answers.
 *
 * @author Damien Carbonne
 */
public interface IssuesAndAnswers {
    /**
     * @return The managed issues.
     */
    public List<Issue> getIssues();

    /**
     * @return The managed answers.
     */
    public Set<? extends IssueAnswer> getAnswers();

    /**
     * @param id The issue id.
     * @return The answer having an {@code id} or {@code null}.
     */
    public Optional<IssueAnswer> getAnswer(IssueId id);

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        final List<Issue> issues = new ArrayList<>();
        final Set<IssueAnswer> answers = new HashSet<>();
        final Map<IssueId, IssueAnswer> issueIdToAnswer = new HashMap<>();

        protected Builder self() {
            return this;
        }

        public Builder issue(Issue issue) {
            Checks.isNotNull(issue, "issue");
            this.issues.add(issue);
            return self();
        }

        public Builder issues(List<? extends Issue> issues) {
            this.issues.addAll(issues);
            return self();
        }

        public Builder answer(IssueAnswer answer) {
            Checks.isNotNull(answer, "answer");
            // We can have only one answer for an given issue
            final IssueAnswer current = issueIdToAnswer.get(answer.getIssueId());
            this.answers.remove(current);
            this.answers.add(answer);
            this.issueIdToAnswer.put(answer.getIssueId(), answer);
            this.answers.add(answer);
            return self();
        }

        public IssuesAndAnswers build() {
            return new IssuesAndAnswerImpl(this);
        }
    }
}

class IssuesAndAnswerImpl implements IssuesAndAnswers {
    private final List<Issue> issues;
    private final Set<IssueAnswer> answers;
    private final Map<IssueId, IssueAnswer> issueIdToAnswer;

    IssuesAndAnswerImpl(Builder builder) {
        this.issues = List.copyOf(builder.issues);
        this.answers = Set.copyOf(builder.answers);
        this.issueIdToAnswer = Map.copyOf(builder.issueIdToAnswer);
    }

    @Override
    public List<Issue> getIssues() {
        return issues;
    }

    @Override
    public Set<IssueAnswer> getAnswers() {
        return answers;
    }

    @Override
    public Optional<IssueAnswer> getAnswer(IssueId id) {
        return Optional.ofNullable(issueIdToAnswer.get(id));
    }
}