package cdc.issues.answers;

import java.time.Instant;
import java.util.ArrayList;
import java.util.List;

import cdc.issues.IssueId;
import cdc.issues.IssueSeverity;
import cdc.issues.Labels;
import cdc.issues.LabelsBuilding;
import cdc.issues.LabelsItem;
import cdc.issues.Metas;
import cdc.issues.MetasBuilding;
import cdc.issues.MetasItem;
import cdc.util.lang.Checks;

/**
 * Interface describing the answer that can be given to an issue.
 *
 * @author Damien Carbonne
 */
public interface IssueAnswer extends MetasItem, LabelsItem {
    /**
     * @return The id of associated issues.<br>
     *         Should <em>NOT</em> be {@code null}.
     */
    public IssueId getIssueId();

    /**
     * @return The author of this answer.
     */
    public String getAuthor();

    /**
     * @return The creation date (as an Instant) of this answer.
     */
    public Instant getCreationDate();

    /**
     * @return The last modification date (as an Instant) of this answer.
     */
    public Instant getModificationDate();

    /**
     * @return The status of this answer.<br>
     *         Should <em>NOT</em> be {@code null}.
     */
    public IssueStatus getStatus();

    /**
     * @return The resolution of this answer.
     *         May be {@code null}.
     */
    public IssueResolution getResolution();

    /**
     * @return The assignee of this answer.
     */
    public String getAssignee();

    /**
     * @return The new severity of this answer.<br>
     *         May be {@code null} if severity was not changed.
     */
    public IssueSeverity getNewSeverity();

    /**
     * @return The comments of this answer.
     */
    public List<? extends IssueComment> getComments();

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder implements MetasBuilding<Builder>, LabelsBuilding<Builder> {
        IssueId issueId;
        String author;
        Instant creationDate;
        Instant modificationDate;
        IssueStatus status = IssueStatus.OPEN;
        IssueResolution resolution = IssueResolution.UNRESOLVED;
        String assignee;
        IssueSeverity newSeverity;
        final List<IssueComment> comments = new ArrayList<>();
        final Metas.Builder metas = Metas.builder();
        Labels labels = Labels.NO_LABELS;

        protected Builder() {
        }

        public Builder issueId(IssueId issueId) {
            this.issueId = issueId;
            return this;
        }

        public Builder author(String author) {
            this.author = author;
            return this;
        }

        public Builder creationDate(Instant creationDate) {
            this.creationDate = creationDate;
            return this;
        }

        public Builder modificationDate(Instant modificationDate) {
            this.modificationDate = modificationDate;
            return this;
        }

        public Builder status(IssueStatus status) {
            this.status = status;
            return this;
        }

        public Builder resolution(IssueResolution resolution) {
            this.resolution = resolution;
            return this;
        }

        public Builder assignee(String assignee) {
            this.assignee = assignee;
            return this;
        }

        public Builder newSeverity(IssueSeverity newSeverity) {
            this.newSeverity = newSeverity;
            return this;
        }

        public Builder comment(IssueComment comment) {
            this.comments.add(comment);
            return this;
        }

        public Builder comments(List<? extends IssueComment> comments) {
            this.comments.addAll(comments);
            return this;
        }

        @Override
        public Builder meta(String name,
                            String value) {
            this.metas.meta(name, value);
            return self();
        }

        @Override
        public Builder meta(String name,
                            String value,
                            String separator) {
            this.metas.meta(name, value, separator);
            return self();
        }

        @Override
        public Builder metas(Metas metas) {
            this.metas.metas(metas);
            return self();
        }

        @Override
        public Builder labels(Labels labels) {
            this.labels = labels;
            return this;
        }

        public IssueAnswer build() {
            final Instant cd = this.creationDate == null ? Instant.now() : this.creationDate;
            return new IssueAnswerImpl(this.issueId,
                                       this.author,
                                       cd,
                                       this.modificationDate == null ? cd : this.modificationDate,
                                       this.status,
                                       this.resolution,
                                       this.assignee,
                                       this.newSeverity,
                                       List.copyOf(this.comments),
                                       this.metas.build(),
                                       this.labels);
        }
    }
}

record IssueAnswerImpl(IssueId issueId,
                       String author,
                       Instant creationDate,
                       Instant modificationDate,
                       IssueStatus status,
                       IssueResolution resolution,
                       String assignee,
                       IssueSeverity newSeverity,
                       List<IssueComment> comments,
                       Metas metas,
                       Labels labels)
        implements IssueAnswer {

    IssueAnswerImpl {
        Checks.isNotNull(issueId, "issueId");
        Checks.isNotNull(status, "status");
        Checks.isNotNull(resolution, "resolution");
    }

    @Override
    public Metas getMetas() {
        return metas;
    }

    @Override
    public Labels getLabels() {
        return labels;
    }

    @Override
    public IssueId getIssueId() {
        return issueId;
    }

    @Override
    public String getAuthor() {
        return author;
    }

    @Override
    public Instant getCreationDate() {
        return creationDate;
    }

    @Override
    public Instant getModificationDate() {
        return modificationDate;
    }

    @Override
    public IssueStatus getStatus() {
        return status;
    }

    @Override
    public IssueResolution getResolution() {
        return resolution;
    }

    @Override
    public String getAssignee() {
        return assignee;
    }

    @Override
    public IssueSeverity getNewSeverity() {
        return newSeverity;
    }

    @Override
    public List<IssueComment> getComments() {
        return comments;
    }
}