package cdc.issues.io;

import java.time.Instant;
import java.util.Optional;

import cdc.issues.IssueUtils;
import cdc.issues.IssuesCountItem;
import cdc.issues.Labels;
import cdc.issues.Metas;
import cdc.issues.Project;
import cdc.issues.Snapshot;
import cdc.issues.rules.Profile;
import cdc.util.lang.Checks;

/**
 * Interface describing the synthetic data associated to a {@link Snapshot}.
 *
 * @author Damien Carbonne
 */
public interface SnapshotData extends IssuesCountItem {
    /**
     * @return The project name.
     */
    public String getProjectName();

    /**
     * @return The project description.
     */
    public String getProjectDescription();

    /**
     * @return The project meta data.
     */
    public Metas getProjectMetas();

    /**
     * @return The project labels.
     */
    public Labels getProjectLabels();

    /**
     * @return The project profile.
     */
    public Optional<? extends Profile> getProfile();

    /**
     * @return The snapshot name.
     */
    public String getSnapshotName();

    /**
     * @return The snapshot description.
     */
    public String getSnapshotDescription();

    /**
     * @return The snapshot meta data.
     */
    public Metas getSnapshotMetas();

    /**
     * @return The snapshot labels.
     */
    public Labels getSnapshotLabels();

    /**
     * @return The snapshot timestamp.
     */
    public Instant getSnapshotTimestamp();

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        String projectName;
        String projectDescription;
        Metas projectMetas = Metas.NO_METAS;
        Labels projectLabels = Labels.NO_LABELS;
        Profile profile;
        String snapshotName;
        String snapshotDescription;
        Metas snapshotMetas = Metas.NO_METAS;
        Labels snapshotLabels = Labels.NO_LABELS;
        Instant snapshotTimestamp = Instant.now();
        int numberOfIssues = -1;
        String issuesHash = "";

        protected Builder self() {
            return this;
        }

        public Builder snapshot(Snapshot snapshot) {
            final Project project = snapshot.getProject().orElse(null);

            if (project != null) {
                projectName(project.getName());
                projectDescription(project.getDescription());
                projectMetas(project.getMetas());
                projectLabels(project.getLabels());
                if (project.getProfile().isPresent()) {
                    profile(project.getProfile().orElseThrow());
                }
            }
            snapshotName(snapshot.getName());
            snapshotDescription(snapshot.getDescription());
            snapshotMetas(snapshot.getMetas());
            snapshotTimestamp(snapshot.getTimestamp());
            numberOfIssues(snapshot.getIssues().size());
            issuesHash(IssueUtils.getHash(snapshot.getIssues()));
            return self();
        }

        public Builder projectName(String name) {
            this.projectName = name;
            return self();
        }

        public Builder projectDescription(String description) {
            this.projectDescription = description;
            return self();
        }

        public Builder projectMetas(Metas metas) {
            this.projectMetas = metas;
            return self();
        }

        public Builder projectLabels(Labels labels) {
            this.projectLabels = labels;
            return self();
        }

        public Builder profile(Profile profile) {
            this.profile = profile;
            return self();
        }

        public Builder snapshotName(String name) {
            this.snapshotName = name;
            return self();
        }

        public Builder snapshotDescription(String description) {
            this.snapshotDescription = description;
            return self();
        }

        public Builder snapshotMetas(Metas metas) {
            this.snapshotMetas = metas;
            return self();
        }

        public Builder snapshotLabels(Labels labels) {
            this.snapshotLabels = labels;
            return self();
        }

        public Builder snapshotTimestamp(Instant timestamp) {
            this.snapshotTimestamp = timestamp;
            return self();
        }

        public Builder numberOfIssues(int number) {
            this.numberOfIssues = number;
            return self();
        }

        public Builder issuesHash(String hash) {
            this.issuesHash = hash;
            return self();
        }

        public SnapshotData build() {
            return new SnapshotDataImpl(this);
        }
    }
}

class SnapshotDataImpl implements SnapshotData {
    private final String projectName;
    private final String projectDescription;
    private final Metas projectMetas;
    private final Labels projectLabels;
    private final Optional<Profile> profile;
    private final String snapshotName;
    private final String snapshotDescription;
    private final Metas snapshotMetas;
    private final Labels snapshotLabels;
    private final Instant snapshotTimestamp;
    private final int numberOfIssues;
    private final String issuesHash;

    SnapshotDataImpl(SnapshotData.Builder builder) {
        this.projectName = builder.projectName;
        this.projectDescription = builder.projectDescription;
        this.projectMetas = Checks.isNotNull(builder.projectMetas, "projectMetas");
        this.projectLabels = Checks.isNotNull(builder.projectLabels, "projectLabels");
        this.profile = Optional.ofNullable(builder.profile);
        this.snapshotName = builder.snapshotName;
        this.snapshotDescription = builder.snapshotDescription;
        this.snapshotMetas = Checks.isNotNull(builder.snapshotMetas, "snapshotMetas");
        this.snapshotLabels = Checks.isNotNull(builder.snapshotLabels, "snapshotLabels");
        this.snapshotTimestamp = Checks.isNotNull(builder.snapshotTimestamp, "snapshotTimestamp");
        this.numberOfIssues = builder.numberOfIssues;
        this.issuesHash = builder.issuesHash;
    }

    @Override
    public String getProjectName() {
        return projectName;
    }

    @Override
    public String getProjectDescription() {
        return projectDescription;
    }

    @Override
    public Metas getProjectMetas() {
        return projectMetas;
    }

    @Override
    public Labels getProjectLabels() {
        return projectLabels;
    }

    @Override
    public Optional<Profile> getProfile() {
        return profile;
    }

    @Override
    public String getSnapshotName() {
        return snapshotName;
    }

    @Override
    public String getSnapshotDescription() {
        return snapshotDescription;
    }

    @Override
    public Metas getSnapshotMetas() {
        return snapshotMetas;
    }

    @Override
    public Labels getSnapshotLabels() {
        return snapshotLabels;
    }

    @Override
    public Instant getSnapshotTimestamp() {
        return snapshotTimestamp;
    }

    @Override
    public int getNumberOfIssues() {
        return numberOfIssues;
    }

    @Override
    public String getIssuesHash() {
        return issuesHash;
    }

    public static Builder builder() {
        return new Builder();
    }
}