package cdc.issues;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Objects;

import cdc.issues.locations.Location;
import cdc.issues.rules.RuleId;
import cdc.util.lang.Checks;

/**
 * Base class used to describe an Issue.
 * <p>
 * <b>Node:</b> This class can be specialized if necessary.
 * <p>
 * An issue may have several targets.<br>
 * For example, when there is a compliance issue between several things,
 * one can not tell which one is at fault.
 *
 * @author Damien Carbonne
 */
public class Issue {
    /**
     * Comparator of Issues sing timestamp.
     */
    public static final Comparator<Issue> TIMESTAMP_COMPARATOR =
            Comparator.comparing(Issue::getTimestamp);

    private final Instant timestamp;
    private final IssueId id;
    private final IssueSeverity severity;
    private final String description;

    /**
     * Creates an Issue.
     *
     * @param timestamp The optional timestamp.
     * @param name The issue name.
     * @param params The parameters.
     * @param locations The target locations.
     * @param severity The issue severity.
     * @param description The issue description.
     */
    protected Issue(Instant timestamp,
                    String domain,
                    String name,
                    Params params,
                    List<? extends Location> locations,
                    IssueSeverity severity,
                    String description) {
        this.timestamp = timestamp == null
                ? Instant.now()
                : timestamp;
        this.id = new IssueId(domain,
                              name,
                              params,
                              locations.toArray(new Location[locations.size()]));
        this.severity = Checks.isNotNull(severity, "severity");
        this.description = Checks.isNotNull(description, "description");
    }

    protected Issue(String domain,
                    String name,
                    Params params,
                    List<? extends Location> locations,
                    IssueSeverity severity,
                    String description) {
        this(null,
             domain,
             name,
             params,
             locations,
             severity,
             description);
    }

    protected Issue(Instant timestamp,
                    String domain,
                    Enum<?> name,
                    Params params,
                    List<? extends Location> locations,
                    IssueSeverity severity,
                    String description) {
        this(timestamp,
             domain,
             name.name(),
             params,
             locations,
             severity,
             description);
    }

    protected Issue(String domain,
                    Enum<?> name,
                    Params params,
                    List<? extends Location> locations,
                    IssueSeverity severity,
                    String description) {
        this(null,
             domain,
             name,
             params,
             locations,
             severity,
             description);
    }

    /**
     * @return The Instant at which this issue was created.
     */
    public final Instant getTimestamp() {
        return timestamp;
    }

    /**
     * @return The id of this issue.
     */
    public IssueId getId() {
        return id;
    }

    /**
     * @return The RuleId of this issue.
     */
    public RuleId getRuleId() {
        return id.getRuleId();
    }

    /**
     * @return The domain of this issue.
     */
    public String getDomain() {
        return id.getDomain();
    }

    /**
     * @return The name of this issue.
     */
    public String getName() {
        return id.getName();
    }

    public <T extends Enum<T>> T getName(Class<T> typeClass) {
        return id.getName(typeClass);
    }

    public Params getParams() {
        return id.getParams();
    }

    /**
     * @return The severity of this issue.
     */
    public final IssueSeverity getSeverity() {
        return severity;
    }

    /**
     * @return The description of this issue.
     */
    public final String getDescription() {
        return description;
    }

    /**
     * @return The target locations of this issue.
     */
    public Location[] getLocations() {
        return id.getLocations();
    }

    /**
     * @return The number of target locations of this issue.
     */
    public final int getNumberOfLocations() {
        return id.getLocations().length;
    }

    public Location getLocationAt(int index) {
        return id.getLocations()[index];
    }

    public <L extends Location> L getLocationAt(int index,
                                                Class<L> cls) {
        return cls.cast(getLocationAt(index));
    }

    @Override
    public int hashCode() {
        return Objects.hash(timestamp,
                            id,
                            severity,
                            description);
    }

    @Override
    public boolean equals(Object object) {
        if (this == object) {
            return true;
        }
        if (!(object instanceof Issue)) {
            return false;
        }
        final Issue other = (Issue) object;
        return Objects.equals(this.timestamp, other.timestamp)
                && Objects.equals(this.id, other.id)
                && this.severity == other.severity
                && Objects.equals(this.description, other.description);
    }

    @Override
    public String toString() {
        return getTimestamp()
                + " - " + getDomain()
                + " - " + getName()
                + " - " + getSeverity()
                + " - " + getDescription()
                + " - " + Arrays.toString(getLocations());
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        protected Instant timestamp = null;
        protected String domain;
        protected String name;
        protected Params params = Params.NO_PARAMS;
        protected IssueSeverity severity;
        protected String description;
        protected final List<Location> locations = new ArrayList<>();

        protected Builder() {
        }

        protected Builder self() {
            return this;
        }

        /**
         * Sets the issue timestamp.
         * <p>
         * <b>WARNING:</b> this should only be used to reconstruct an issue -from a file, stream, ...).
         *
         * @param timestamp The timestamp.
         * @return This builder.
         */
        public Builder timestamp(Instant timestamp) {
            this.timestamp = timestamp;
            return self();
        }

        /**
         * Sets the issue RuleId.
         * <p>
         * This is equivalent to setting its domain and name.
         *
         * @param ruleId The rule id.
         * @return This builder.
         */
        public Builder ruleId(RuleId ruleId) {
            this.domain = ruleId.getDomain();
            this.name = ruleId.getName();
            return self();
        }

        /**
         * Sets the issue domain.
         *
         * @param domain The domain.
         * @return This builder.
         */
        public Builder domain(String domain) {
            this.domain = domain;
            return self();
        }

        /**
         * Sets the issue name.
         *
         * @param name The name.
         * @return This builder.
         */
        public Builder name(String name) {
            this.name = name;
            return self();
        }

        /**
         * Sets the issue name and its severity if the {@code name} implements
         * {@link IssueSeverityItem} and current severity is {@code null}.
         *
         * @param name The name.
         * @return This builder.
         */
        public Builder name(Enum<?> name) {
            this.name = name.name();
            if (severity == null && name instanceof IssueSeverityItem) {
                severity(((IssueSeverityItem) name).getSeverity());
            }
            return self();
        }

        /**
         * Sets the effective parameters that have been used to configure the issue detection.
         *
         * @param params The parameters.
         * @return This builder.
         */
        public Builder params(Params params) {
            this.params = params;
            return self();
        }

        /**
         * Adds an issue location.
         *
         * @param location The location.
         * @return This builder.
         */
        public Builder addLocation(Location location) {
            this.locations.add(location);
            return self();
        }

        /**
         * Sets the issue locations.
         *
         * @param locations The locations
         * @return This builder.
         */
        public Builder locations(Location... locations) {
            this.locations.clear();
            Collections.addAll(this.locations, locations);
            return this;
        }

        /**
         * Sets the issue locations.
         *
         * @param locations The locations
         * @return This builder.
         */
        public Builder locations(List<Location> locations) {
            this.locations.clear();
            this.locations.addAll(locations);
            return this;
        }

        /**
         * Sets the issue severity.
         *
         * @param severity The severity.
         * @return This builder.
         */
        public Builder severity(IssueSeverity severity) {
            this.severity = severity;
            return self();
        }

        /**
         * Sets the issue description.
         *
         * @param description The description.
         * @return This builder.
         */
        public Builder description(String description) {
            this.description = description;
            return self();
        }

        public Issue build() {
            return new Issue(timestamp,
                             domain,
                             name,
                             params,
                             locations,
                             severity,
                             description);
        }
    }

    public abstract static class AbstractIssueBuilder<B extends AbstractIssueBuilder<B, I>, I extends Issue> extends Builder {
        protected AbstractIssueBuilder() {
        }

        @Override
        protected abstract B self();

        @Override
        public Builder timestamp(Instant timestamp) {
            super.timestamp(timestamp);
            return self();
        }

        @Override
        public B ruleId(RuleId ruleId) {
            super.ruleId(ruleId);
            return self();
        }

        @Override
        public B domain(String domain) {
            super.domain(domain);
            return self();
        }

        @Override
        public B name(String name) {
            super.name(name);
            return self();
        }

        @Override
        public B name(Enum<?> name) {
            super.name(name);
            return self();
        }

        @Override
        public B params(Params params) {
            super.params(params);
            return self();
        }

        @Override
        public B addLocation(Location location) {
            super.addLocation(location);
            return self();
        }

        @Override
        public B locations(Location... locations) {
            super.locations(locations);
            return self();
        }

        @Override
        public B locations(List<Location> locations) {
            super.locations(locations);
            return self();
        }

        @Override
        public B severity(IssueSeverity severity) {
            super.severity(severity);
            return self();
        }

        @Override
        public B description(String description) {
            super.description(description);
            return self();
        }

        @Override
        public abstract I build();
    }
}