package cdc.issues;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import cdc.util.lang.Checks;

/**
 * A collector of issues that can delegate issues.
 *
 * @author Damien Carbonne
 * @param <I> The issue type.
 */
public class IssuesCollector<I extends Issue> implements IssuesHandler<I> {
    private final List<I> issues = new ArrayList<>();
    private final IssuesHandler<? super I> delegate;

    public IssuesCollector(IssuesHandler<? super I> delegate) {
        this.delegate = Checks.isNotNull(delegate, "delegate");
    }

    public IssuesCollector() {
        this(IssuesHandler.VOID);
    }

    /**
     * @return The delegate (possibly {@code null}).
     */
    public IssuesHandler<? super I> getDelegate() {
        return delegate;
    }

    /**
     * Clears collected issues.
     */
    public void clear() {
        issues.clear();
    }

    /**
     * @return The list of collected issues.
     */
    public List<I> getIssues() {
        return issues;
    }

    /**
     * Returns a list of issues having a severity.
     *
     * @param severity The severity.
     * @return A list of issues that have a certain {@code severity}.
     */
    public List<I> getIssues(IssueSeverity severity) {
        return issues.stream()
                     .filter(i -> i.getSeverity() == severity)
                     .collect(Collectors.toList());
    }

    @Override
    public void issue(I issue) {
        issues.add(issue);
        delegate.issue(issue);
    }
}