package cdc.issues.locations;

import cdc.util.strings.StringUtils;

/**
 * Base interface used to describe a location inside a resource.
 * <p>
 * It is composed of 2 parts:
 * <ul>
 * <li>external: the path to a resource,
 * <li>internal: the anchor that optionally designates something inside the resource.
 * </ul>
 *
 * @author Damien Carbonne
 */
public interface Location {
    public static final Location UNDEFINED = new DefaultLocation("", null);

    /**
     * @return The external part of the location, the path of the designated resource.<br>
     *         It is typically a file name, an URI, a kind of 'absolute' identifier.
     */
    public String getPath();

    /**
     * @return The internal part of the location, inside the resource.
     *         It may be {@code null}.
     */
    public String getAnchor();

    /**
     * @return {@code true} if this location has a meaningful anchor.
     */
    public default boolean hasAnchor() {
        return !StringUtils.isNullOrEmpty(getAnchor());
    }

    public static String toString(Location location) {
        final String anchor = location.getAnchor();
        return StringUtils.isNullOrEmpty(anchor)
                ? location.getPath()
                : location.getPath() + ":" + anchor;
    }
}