package cdc.issues.locations;

import cdc.util.lang.Checks;

/**
 * Implementation of Location dedicated to workbooks.
 *
 * @author Damien Carbonne
 */
public class WorkbookLocation extends AbstractLocation {
    private final String systemId;
    private final String sheetName;
    private final int rowNumber;
    private final String columnName;

    private WorkbookLocation(String systemId,
                             String sheetName,
                             int rowNumber,
                             String columnName) {
        this.systemId = Checks.isNotNull(systemId, "systemId");
        this.sheetName = sheetName;
        this.rowNumber = rowNumber;
        this.columnName = columnName;
        if ((rowNumber > 0 || columnName != null) && sheetName == null) {
            throw new IllegalArgumentException("Sheet name is missing.");
        }
    }

    @Override
    public String getPath() {
        return getSystemId();
    }

    @Override
    public String getAnchor() {
        if (sheetName == null) {
            return "";
        } else {
            if (columnName == null) {
                if (rowNumber <= 0) {
                    return sheetName;
                } else {
                    return sheetName + "[:" + rowNumber + "]";
                }
            } else {
                if (rowNumber <= 0) {
                    return sheetName + "[" + columnName + "]";
                } else {
                    return sheetName + "[" + columnName + ":" + rowNumber + "]";
                }
            }
        }
    }

    public String getSystemId() {
        return systemId;
    }

    public String getSheetName() {
        return sheetName;
    }

    public int getRowNumber() {
        return rowNumber;
    }

    public String getColumnName() {
        return columnName;
    }

    public boolean isColumnLocation() {
        return rowNumber <= 0 && columnName != null;
    }

    public boolean isRowLocation() {
        return rowNumber > 0 && columnName == null;
    }

    public boolean isCellLocation() {
        return rowNumber > 0 && columnName != null;
    }

    public boolean isSheetLocation() {
        return sheetName != null && rowNumber <= 0 && columnName == null;
    }

    @Override
    public int hashCode() {
        return super.hashCode();
    }

    @Override
    public boolean equals(Object object) {
        return super.equals(object);
    }

    public static Builder builder() {
        return new Builder();
    }

    public static class Builder {
        private String systemId;
        private String sheetName = null;
        private int rowNumber = 0;
        private String columnName = null;

        private Builder() {
            super();
        }

        public Builder systemId(String systemId) {
            this.systemId = systemId;
            return this;
        }

        public Builder sheetName(String sheetName) {
            this.sheetName = sheetName;
            return this;
        }

        public Builder rowNumber(int rowNumber) {
            this.rowNumber = rowNumber;
            return this;
        }

        public Builder columnName(String columnName) {
            this.columnName = columnName;
            return this;
        }

        public WorkbookLocation build() {
            return new WorkbookLocation(systemId,
                                        sheetName,
                                        rowNumber,
                                        columnName);
        }
    }
}