package cdc.issues.rules;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import cdc.issues.rules.IssuesDetector.Descriptor;
import cdc.tuples.Tuple2;
import cdc.util.lang.Checks;
import cdc.util.lang.Introspection;

/**
 * Default implementation of {@link RulesCatalog}.
 *
 * @author Damien Carbonne
 */
public class DefaultRulesCatalog implements RulesCatalog {
    /**
     * (domain -> Set[Rule]) map.
     */
    private final Map<String, Set<Rule>> domainToRules = new HashMap<>();

    /**
     * Set of Rules.
     */
    private final Set<Rule> rules = new HashSet<>();

    /**
     * Set of descriptors.
     */
    private final Set<IssuesDetector.Descriptor<?>> descriptors = new HashSet<>();

    /**
     * ((Rule, Data class) -> Descriptor) map.
     */
    private final Map<Tuple2<Rule, Class<?>>, IssuesDetector.Descriptor<?>> ruleDataClassToDescriptor = new HashMap<>();

    /**
     * (dataClass -> Set<Descriptor>) map.
     */
    private final Map<Class<?>, Set<IssuesDetector.Descriptor<?>>> dataClassToDescriptors = new HashMap<>();

    public DefaultRulesCatalog() {
        super();
    }

    public DefaultRulesCatalog register(Rule rule) {
        Checks.isNotNull(rule, "rule");
        Checks.doesNotContain(rules, rule, "rules");

        rules.add(rule);
        final Set<Rule> set = domainToRules.computeIfAbsent(rule.getId().getDomain(), k -> new HashSet<>());
        set.add(rule);
        return this;
    }

    public <T> DefaultRulesCatalog register(IssuesDetector.Descriptor<T> descriptor) {
        Checks.isNotNull(descriptor, "descriptor");
        Checks.doesNotContain(descriptors, descriptor, "descriptors");

        descriptors.add(descriptor);
        final Set<IssuesDetector.Descriptor<?>> set =
                dataClassToDescriptors.computeIfAbsent(descriptor.getDataClass(), k -> new HashSet<>());
        set.add(descriptor);

        for (final Rule rule : descriptor.getRules()) {
            final Tuple2<Rule, Class<?>> key = new Tuple2<>(rule, descriptor.getDataClass());
            if (!hasRule(rule)) {
                register(rule);
            }
            if (ruleDataClassToDescriptor.containsKey(key)) {
                throw new IllegalArgumentException("Duplicate descriptor for " + key);
            } else {
                ruleDataClassToDescriptor.put(key, descriptor);
            }
        }
        return this;
    }

    @Override
    public Set<String> getDomains() {
        return domainToRules.keySet();
    }

    @Override
    public Set<Rule> getRules() {
        return rules;
    }

    @Override
    public Set<Rule> getRules(String domain) {
        return domainToRules.getOrDefault(domain, Collections.emptySet());
    }

    @Override
    public Set<Descriptor<?>> getDescriptors() {
        return descriptors;
    }

    @Override
    public <T> Set<Descriptor<T>> getDescriptors(Class<T> dataClass) {
        final Set<IssuesDetector.Descriptor<?>> set =
                dataClassToDescriptors.getOrDefault(dataClass, Collections.emptySet());
        return Introspection.uncheckedCast(set);
    }

    @Override
    public <T> Descriptor<T> getDescriptorOrNull(Rule rule,
                                                 Class<T> dataClass) {
        final Tuple2<Rule, Class<?>> key = new Tuple2<>(rule, dataClass);
        final IssuesDetector.Descriptor<T> descriptor =
                Introspection.uncheckedCast(ruleDataClassToDescriptor.get(key));
        return descriptor;
    }
}