package cdc.issues.rules;

import java.util.Collections;
import java.util.Set;

import cdc.issues.FormalParams;
import cdc.issues.Issue;
import cdc.issues.IssuesHandler;
import cdc.issues.Params;
import cdc.issues.locations.Location;
import cdc.util.lang.CollectionUtils;

/**
 * Interface implemented by classes that can detect issues.
 *
 * @author Damien Carbonne
 *
 * @param <T> The data type onto which issue can be detected.
 */
public interface IssuesDetector<T> {

    /**
     * Interface implemented by classes that can describe and create an IssueDetector.
     *
     * @author Damien Carbonne
     *
     * @param <T> The data type onto which issue can be detected.
     */
    public interface Descriptor<T> {
        /**
         * @return The class of data that can be analyzed.
         */
        public Class<T> getDataClass();

        /**
         * @return The set of all supported Rules.
         */
        public Set<Rule> getRules();

        /**
         * @return The description of parameters that can or must be
         *         used to create an associated IssuesDetector.
         */
        public FormalParams getFormalParams();

        /**
         * Creates and configures an IssuesDetector to analyze some rules.
         *
         * @param params The effective parameters to use.
         *            Meaningless parameters can be passed. They musty be ignored.
         * @param rules The rules for which analysis must be performed.<br>
         *            It must be a subset of supported rules.
         * @return A newly created IssuesDetector, configured with {@code params} and {@code rules}.
         */
        public IssuesDetector<T> create(Params params,
                                        Set<Rule> rules);
    }

    public static abstract class AbstractDescriptor<T> implements Descriptor<T> {
        private final Class<T> dataClass;
        private final FormalParams formalParams;
        private final Set<Rule> rules;

        protected AbstractDescriptor(Class<T> dataClass,
                                     FormalParams formalParams,
                                     Rule... rules) {
            this.dataClass = dataClass;
            this.formalParams = formalParams;
            this.rules = Collections.unmodifiableSet(CollectionUtils.toSet(rules));
        }

        protected AbstractDescriptor(Class<T> dataClass,
                                     Rule... rules) {
            this(dataClass,
                 FormalParams.NO_PARAMS,
                 rules);
        }

        @Override
        public final Class<T> getDataClass() {
            return dataClass;
        }

        @Override
        public final Set<Rule> getRules() {
            return rules;
        }

        @Override
        public final FormalParams getFormalParams() {
            return formalParams;
        }
    }

    /**
     * @return The descriptor that was used to create this IssuesDetector.
     */
    public Descriptor<T> getDescriptor();

    /**
     * @return The class of data that can be analyzed.
     */
    public default Class<T> getDataClass() {
        return getDescriptor().getDataClass();
    }

    /**
     * @return The set of enabled Rules.
     */
    public Set<Rule> getEnabledRules();

    /**
     * @return The parameters used to configure this detector.
     */
    public Params getParams();

    /**
     * Analyzes one data.
     *
     * @param data The data to analyze.
     * @param location The data location.
     * @param issuesHandler The issues handler that must be used to store issues.
     */
    public void analyze(T data,
                        Location location,
                        IssuesHandler<Issue> issuesHandler);

    public static String toString(IssuesDetector<?> detector) {
        final StringBuilder builder = new StringBuilder();
        builder.append("IssuesDetector<")
               .append(detector.getDataClass().getSimpleName())
               .append(">(");
        boolean first = true;
        for (final Rule rule : detector.getEnabledRules()) {
            if (first) {
                first = false;
            } else {
                builder.append(",");
            }
            builder.append(rule.getName());
        }
        builder.append(")");
        return builder.toString();
    }
}