package cdc.issues.impl;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import cdc.issues.Params;
import cdc.issues.rules.ConfiguredRule;
import cdc.issues.rules.Profile;
import cdc.issues.rules.Rule;
import cdc.util.lang.Checks;

/**
 * Default implementation of {@link Profile}.
 *
 * @author Damien Carbonne
 */
public class ProfileImpl implements Profile {
    private static final String METAS = "metas";
    private static final String PARAMS = "params";
    private static final String RULE = "rule";

    /** The profile name. */
    private String name;
    /** The profile meta datas. */
    private Params metas = Params.NO_PARAMS;
    /** The set of enabled rules. */
    private final Set<Rule> enabledRules = new HashSet<>();
    /** The declared rules and their associated parameters. */
    private final Map<Rule, Params> ruleToParams = new HashMap<>();

    public ProfileImpl(String name) {
        this.name = name;
    }

    public ProfileImpl setName(String name) {
        this.name = name;
        return this;
    }

    public ProfileImpl setMetas(Params metas) {
        this.metas = Checks.isNotNull(metas, METAS);
        return this;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public Params getMetas() {
        return metas;
    }

    /**
     * Adds a rule with parameters and enable it.
     *
     * @param rule The rule.
     * @param params The parameters.
     * @return This object.
     */
    public ProfileImpl add(Rule rule,
                           Params params) {
        Checks.isNotNull(rule, RULE);
        Checks.isNotNull(params, PARAMS);

        enabledRules.add(rule);
        ruleToParams.put(rule, params);
        return this;
    }

    /**
     * Adds a rule with no parameters, and enables it.
     *
     * @param rule The rule.
     * @return This object.
     */
    public ProfileImpl add(Rule rule) {
        return add(rule, Params.NO_PARAMS);
    }

    /**
     * Removes a rule.
     *
     * @param rule The rule.
     * @return This object.
     */
    public ProfileImpl remove(Rule rule) {
        ruleToParams.remove(rule);
        enabledRules.remove(rule);
        return this;
    }

    /**
     * Enables or disables a rule.
     * <p>
     * If the rule is not declared, declares it.
     *
     * @param rule The rule.
     * @param enabled {@code true} if the rule must be enabled.
     * @return This object.
     */
    public ProfileImpl setEnabled(Rule rule,
                                  boolean enabled) {
        Checks.isNotNull(rule, RULE);

        if (enabled) {
            ruleToParams.putIfAbsent(rule, Params.NO_PARAMS);
            enabledRules.add(rule);
        } else {
            enabledRules.remove(rule);
        }
        return this;
    }

    @Override
    public Set<Rule> getRules() {
        return ruleToParams.keySet();
    }

    @Override
    public Set<ConfiguredRule> getConfiguredRules() {
        final Set<ConfiguredRule> set = new HashSet<>();
        for (final Map.Entry<Rule, Params> entry : ruleToParams.entrySet()) {
            set.add(new ConfiguredRule(entry.getKey(), entry.getValue()));
        }
        return set;
    }

    @Override
    public Set<Rule> getEnabledRules() {
        return enabledRules;
    }

    @Override
    public Set<ConfiguredRule> getEnabledConfiguredRules() {
        final Set<ConfiguredRule> set = new HashSet<>();
        for (final Map.Entry<Rule, Params> entry : ruleToParams.entrySet()) {
            if (isEnabled(entry.getKey())) {
                set.add(new ConfiguredRule(entry.getKey(), entry.getValue()));
            }
        }
        return set;
    }

    @Override
    public boolean isEnabled(Rule rule) {
        Checks.isNotNull(rule, RULE);
        return enabledRules.contains(rule);
    }

    @Override
    public Params getParams(Rule rule) {
        Checks.isNotNull(rule, RULE);
        return ruleToParams.get(rule);
    }
}