package cdc.issues.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import cdc.issues.rules.IssuesDetector;
import cdc.issues.rules.IssuesDetector.Factory;
import cdc.issues.rules.Rule;
import cdc.issues.rules.RuleId;
import cdc.issues.rules.RulesCatalog;
import cdc.tuples.Tuple2;
import cdc.util.lang.Checks;
import cdc.util.lang.Introspection;

/**
 * Default implementation of {@link RulesCatalog}.
 *
 * @author Damien Carbonne
 */
public class RulesCatalogImpl implements RulesCatalog {
    /**
     * (domain -> Set[Rule]) map.
     */
    private final Map<String, Set<Rule>> domainToRules = new HashMap<>();

    /**
     * Set of Rules.
     */
    private final Set<Rule> rules = new HashSet<>();

    private final Map<RuleId, Rule> idToRules = new HashMap<>();

    /**
     * Set of factories.
     */
    private final Set<IssuesDetector.Factory<?>> factories = new HashSet<>();

    /**
     * ((Rule, Data class) -> Descriptor) map.
     */
    private final Map<Tuple2<Rule, Class<?>>, IssuesDetector.Factory<?>> ruleDataClassToFactory = new HashMap<>();

    /**
     * (dataClass -> Set<Descriptor>) map.
     */
    private final Map<Class<?>, Set<IssuesDetector.Factory<?>>> dataClassToFactories = new HashMap<>();

    public RulesCatalogImpl() {
        super();
    }

    public RulesCatalogImpl register(Rule rule) {
        Checks.isNotNull(rule, "rule");
        Checks.doesNotContain(rules, rule, "rules");
        Checks.doesNotContainKey(idToRules, rule.getId(), "idToRules");

        rules.add(rule);
        idToRules.put(rule.getId(), rule);
        final Set<Rule> set = domainToRules.computeIfAbsent(rule.getId().getDomain(), k -> new HashSet<>());
        set.add(rule);
        return this;
    }

    public <T> RulesCatalogImpl register(IssuesDetector.Factory<T> factory) {
        Checks.isNotNull(factory, "factory");
        Checks.doesNotContain(factories, factory, "factory");

        factories.add(factory);
        final Set<IssuesDetector.Factory<?>> set =
                dataClassToFactories.computeIfAbsent(factory.getDataClass(), k -> new HashSet<>());
        set.add(factory);

        for (final Rule rule : factory.getSupportedRules()) {
            final Tuple2<Rule, Class<?>> key = new Tuple2<>(rule, factory.getDataClass());
            if (!hasRule(rule)) {
                register(rule);
            }
            if (ruleDataClassToFactory.containsKey(key)) {
                throw new IllegalArgumentException("Duplicate factory for " + key);
            } else {
                ruleDataClassToFactory.put(key, factory);
            }
        }
        return this;
    }

    @Override
    public Set<String> getDomains() {
        return domainToRules.keySet();
    }

    @Override
    public Set<Rule> getRules() {
        return rules;
    }

    @Override
    public Set<Rule> getRules(String domain) {
        return domainToRules.getOrDefault(domain, Collections.emptySet());
    }

    @Override
    public Optional<Rule> getRule(RuleId id) {
        return Optional.ofNullable(idToRules.get(id));
    }

    @Override
    public Set<Factory<?>> getFactories() {
        return factories;
    }

    @Override
    public <T> Set<Factory<T>> getFactories(Class<T> dataClass) {
        final Set<IssuesDetector.Factory<?>> set =
                dataClassToFactories.getOrDefault(dataClass, Collections.emptySet());
        return Introspection.uncheckedCast(set);
    }

    @Override
    public <T> Optional<Factory<T>> getFactory(Rule rule,
                                               Class<T> dataClass) {
        final Tuple2<Rule, Class<?>> key = new Tuple2<>(rule, dataClass);
        final Factory<T> value = Introspection.uncheckedCast(ruleDataClassToFactory.get(key));
        return Optional.ofNullable(value);
    }
}