package cdc.util.cli;

import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Stream;

import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.OptionGroup;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.Logger;

import cdc.util.files.Files;
import cdc.util.lang.Checks;
import cdc.util.lang.ExceptionWrapper;

/**
 * Utility that can be used to create main programs.
 * <p>
 * The purpose is to force the adoption of a common pattern for such classes.
 * <p>
 * Some options are automatically defined and handled:
 *
 * <ul>
 * <li><b>help:</b> if passed, help is printed and program is exited.
 *
 * <li><b>version:</b> if passed, version is printed and program is exited.<br>
 * Override {@link AbstractMainSupport#getVersion()} to define version.
 *
 * <li><b>args-list &lt;FILE&gt;:</b> if passed, command line arguments
 * are read from the designated file and added to other command line arguments.<br>
 * This is useful when command line is too long.<br>
 * Override {@link AbstractMainSupport#addArgsFileOption(Options)} to force its presence.
 * </ul>
 *
 * <b>Args file syntax</b><br>
 * The file must contains one argument (option of value) per line.<br>
 * A line that is empty or starts with any number of white spaces followed by '#' and any other characters is a ignored.<br>
 * Exemple:<br>
 * <pre>
 * # (ignored)
 * --option1
 * --option2
 * --option3
 * value1
 * value2
 *    # (ignored)
 * value3
 * </pre>
 *
 * @author Damien Carbonne
 *
 * @param <A> MainArgs type.
 * @param <R> Returned type.
 */
public abstract class AbstractMainSupport<A, R> {
    private final Class<?> mainClass;
    private final Logger logger;
    private Result result = Result.SUCCESS;

    public static final String DEFAULT_PARTS_SEPARATOR = "::";

    public static final String ARGS_FILE = "args-file";
    public static final String CHARSET = "charset";
    public static final String DRIVER = "driver";
    public static final String HELP = "help";
    public static final String INPUT = "input";
    public static final String INPUT_DIR = "input-dir";
    public static final String OUTPUT = "output";
    public static final String OUTPUT_DIR = "output-dir";
    public static final String PASSWORD = "password";
    public static final String PATH = "path";
    public static final String PREFIX = "prefix";
    public static final String TMP_DIR = "tmp-dir";
    public static final String URL = "url";
    public static final String USER = "user";
    public static final String VERSION = "version";

    public enum Result {
        COMMAND_LINE_ERROR,
        EXECUTION_ERROR,
        SUCCESS
    }

    public enum ExceptionThrowing {
        NEVER,
        EXECUTION,
        COMMAND_LINE_AND_EXECUTION;

        public boolean matchesExecution() {
            return this == EXECUTION || this == COMMAND_LINE_AND_EXECUTION;
        }

        public boolean matchesCommandLine() {
            return this == COMMAND_LINE_AND_EXECUTION;
        }
    }

    protected AbstractMainSupport(Class<?> mainClass,
                                  Logger logger) {
        this.mainClass = mainClass;
        this.logger = logger;
    }

    /**
     * Returns {@code true} if {@link #ARGS_FILE} option must be added.
     * <p>
     * The default implementation returns {@code true} if one option
     * contains multiple values.
     *
     * @param options The options.
     * @return {@code true} if {@link #ARGS_FILE} option must be added.
     */
    protected boolean addArgsFileOption(Options options) {
        return hasMultipleValuesOption(options);
    }

    protected void addStandardOptions(Options options) {
        final Option help = Option.builder("h")
                                  .longOpt(HELP)
                                  .desc("Prints this help and exits.")
                                  .build();
        options.addOption(help);

        if (getVersion() != null) {
            final Option version = Option.builder("v")
                                         .longOpt(VERSION)
                                         .desc("Prints version and exits.")
                                         .build();
            options.addOption(version);

            final OptionGroup group = new OptionGroup();
            group.addOption(help);
            group.addOption(version);
            options.addOptionGroup(group);
        }
        if (addArgsFileOption(options)) {
            options.addOption(Option.builder()
                                    .longOpt(ARGS_FILE)
                                    .desc("Name of the file from which options can be read.\n"
                                            + "A line is either ignored or interpreted as a single argument (option or value).\n"
                                            + "A line is ignored when it is empty or starts by any number of white spaces followed by '#'.\n"
                                            + "A line that only contains white spaces is an argument.")
                                    .hasArg()
                                    .build());
        }
    }

    protected final Logger getLogger() {
        return logger;
    }

    /**
     * Returns the version.
     * <p>
     * If {@code null} is returned (default implementation), no version option is available.
     *
     * @return The version.
     */
    protected String getVersion() {
        return null;
    }

    /**
     * Returns the help header.
     * <p>
     * If {@code null} is returned (default implementation), no help header is printed.
     *
     * @return The help header.
     */
    protected String getHelpHeader() {
        return null;
    }

    /**
     * Returns the help footer.
     * <p>
     * If {@code null} is returned (default implementation), no help footer is printed.
     *
     * @return The help footer.
     */
    protected String getHelpFooter() {
        return null;
    }

    /**
     * Creates specific options and add them to an options collection.
     * <p>
     * The standard options must not be added.
     *
     * @param options The options.
     */
    protected abstract void addSpecificOptions(Options options);

    /**
     * Analyzes the command line.
     * <p>
     * These options are already handled:
     * <ul>
     * <li>help
     * <li>version
     * <li>args-file
     * </ul>
     *
     * @param cl The command line.
     * @return A MainARgs instance.
     * @throws ParseException When command line parsing has a problem.
     */
    protected abstract A analyze(CommandLine cl) throws ParseException;

    /**
     * Executes the main program.
     *
     * @param margs The main arguments.
     * @return The optional result.
     * @throws Exception When program has a problem.
     */
    protected abstract R execute(A margs) throws Exception;

    public static <E extends Enum<E> & OptionEnum> void addNoArgOptions(Options options,
                                                                        Class<E> enumClass) {
        for (final E e : enumClass.getEnumConstants()) {
            options.addOption(Option.builder(e.getShortName())
                                    .longOpt(e.getName())
                                    .desc(e.getDescription())
                                    .build());
        }
    }

    public static <E extends Enum<E> & OptionEnum> void addGroupedNoArgOptions(Options options,
                                                                               Class<E> enumClass,
                                                                               boolean required) {
        addNoArgOptions(options, enumClass);
        createGroup(options, required, enumClass.getEnumConstants());
    }

    @SafeVarargs
    public static <E extends Enum<E> & OptionEnum> OptionGroup createGroup(Options options,
                                                                           boolean required,
                                                                           E... values) {
        final OptionGroup group = new OptionGroup();
        for (final E value : values) {
            group.addOption(options.getOption(value.getName()));
        }
        options.addOptionGroup(group);
        group.setRequired(required);
        return group;
    }

    @SafeVarargs
    public static <E extends Enum<E> & OptionEnum> OptionGroup createGroup(Options options,
                                                                           E... values) {
        return createGroup(options, false, values);
    }

    @FunctionalInterface
    public static interface Maskable<E extends Enum<E>> {
        public void setEnabled(E e,
                               boolean enabled);
    }

    public static <E extends Enum<E> & OptionEnum> void setMask(CommandLine cl,
                                                                Class<E> enumClass,
                                                                Maskable<E> maskable) {
        for (final E e : enumClass.getEnumConstants()) {
            maskable.setEnabled(e, cl.hasOption(e.getName()));
        }
    }

    /**
     * Removes the comment part of a string.
     * <p>
     * A comment starts with any number of white spaces followed by '#'
     *
     * @param s The string
     * @return {@code s} with comments removed.
     */
    private static String removeComments(String s) {
        final int pos = s.indexOf('#');
        if (pos < 0) {
            return s;
        } else {
            // Part of s that is before '#'
            final String k = s.substring(0, pos);
            // TODO replace with isBlank() when using Java 11
            if (k.trim().isEmpty()) {
                return "";
            } else {
                // TODO remove trailing spaces using stripTrailing when using Java 11
                return k;
            }
        }
    }

    private static String[] loadArgs(File file) throws IOException {
        try (final InputStream in = new BufferedInputStream(new FileInputStream(file));
                final BufferedReader reader = new BufferedReader(new InputStreamReader(in))) {
            final List<String> list = new ArrayList<>();
            String line = null;
            while ((line = reader.readLine()) != null) {
                line = removeComments(line);
                if (!line.isEmpty()) {
                    list.add(line);
                }
            }
            return list.toArray(new String[list.size()]);
        }
    }

    /**
     * Default main program.
     * <p>
     * This does the following things:
     * <ol>
     * <li>Build options.
     * <li>Parse the command line strings and creates a CommandLine instance.
     * <li>Analyze the CommandLine instance and create a MainArgs instance.
     * <li>Calls execute with the MainARgs instance.
     * </ol>
     *
     * @param args The command line arguments.
     * @return The optional result.
     */
    public R main(String[] args) {
        // Build all options as declared
        final Options allOptions = buildOptions();
        // Build all options as optional
        final Options optionalOptions = getOptionsAsOptional();
        final DefaultParser parser = new DefaultParser();
        Exception x = null;
        try {
            final String[] effectiveArgs;
            // Analysis of optional args file
            final CommandLine cl0 = parser.parse(optionalOptions, args);
            final CommandLine cl1;
            final File argsFile = getValueAsNullOrExistingFile(cl0, ARGS_FILE, null);
            if (argsFile != null) {
                final String[] fileArgs = loadArgs(argsFile);
                // Concatenate file args followed by normal args
                effectiveArgs = Stream.concat(Arrays.stream(fileArgs), Arrays.stream(args))
                                      .toArray(String[]::new);
                cl1 = parser.parse(optionalOptions, effectiveArgs);
            } else {
                effectiveArgs = args;
                cl1 = cl0;
            }

            // First parsing and processing
            if (cl1.hasOption(HELP)) {
                printHelp(allOptions, null);
                return null;
            }
            if (cl1.hasOption(VERSION)) {
                printVersion();
                return null;
            }

            // Normal parsing and processing
            final CommandLine cl2 = parser.parse(allOptions, effectiveArgs);
            final A margs = analyze(cl2);
            try {
                return execute(margs);
            } catch (final Exception e) {
                result = Result.EXECUTION_ERROR;
                logger.catching(e);
                x = e;
            }
        } catch (final Exception e) {
            result = Result.COMMAND_LINE_ERROR;
            printHelp(allOptions, e);
            x = e;
        }

        // Here x != null
        if (result == Result.EXECUTION_ERROR) {
            if (getExceptionThrowing().matchesExecution()) {
                throw ExceptionWrapper.wrap(x);
            } else {
                return null;
            }
        } else if (result == Result.COMMAND_LINE_ERROR) {
            if (getExceptionThrowing().matchesCommandLine()) {
                throw ExceptionWrapper.wrap(x);
            } else {
                return null;
            }
        } else {
            return null;
        }
    }

    protected final Result getResult() {
        return result;
    }

    /**
     * @return When exceptions shall be thrown.
     *         Default to {@link ExceptionThrowing#EXECUTION}.
     */
    protected ExceptionThrowing getExceptionThrowing() {
        return ExceptionThrowing.EXECUTION;
    }

    /**
     * @param options The options.
     * @return {@code true} if {@code options} contains an options having multiple values.
     */
    private static boolean hasMultipleValuesOption(Options options) {
        for (final Option option : options.getOptions()) {
            if (option.hasArgs()) {
                return true;
            }
        }
        return false;
    }

    /**
     * Creates options, add standard and specific options and returns them.
     *
     * @return The options.
     */
    protected Options buildOptions() {
        final Options options = new Options();
        addSpecificOptions(options);
        addStandardOptions(options);
        return options;
    }

    /**
     * @return Build all options and make them optional.
     */
    protected Options getOptionsAsOptional() {
        final Options options = new Options();
        for (final Option option : buildOptions().getOptions()) {
            option.setRequired(false);
            options.addOption(option);
        }
        return options;
    }

    protected void printHelp(Options options,
                             Exception e) {
        final HelpFormatter formatter = new HelpFormatter();
        final StringBuilder header = new StringBuilder();
        if (getHelpHeader() != null) {
            header.append("\n");
            header.append(getHelpHeader());
        }
        header.append("\n");
        if (e != null && e.getMessage() != null) {
            header.append(e.getMessage());
            header.append("\n");
        }

        formatter.printHelp(mainClass.getSimpleName(),
                            header.toString(),
                            options,
                            getHelpFooter(),
                            true);
        if (e != null) {
            if (e instanceof ParseException) {
                if (logger.isEnabled(Level.DEBUG)) {
                    logger.catching(e);
                }
            } else {
                logger.catching(e);
            }
        }
    }

    protected void printVersion() {
        System.out.println(getVersion());
    }

    private static ParseException invalidArg(CommandLine cl,
                                             String opt,
                                             String message) {
        final StringBuilder builder = new StringBuilder();
        builder.append("Invalid arg for ");
        builder.append(opt);
        builder.append(" args:");
        final String[] values = cl.getOptionValues(opt);
        if (values != null) {
            final int count = Math.min(5, values.length);
            for (int index = 0; index < count; index++) {
                builder.append(" '");
                builder.append(values[index]);
                builder.append('\'');
            }
            if (count < values.length) {
                builder.append(" ...");
            }
        }
        if (message != null) {
            builder.append(" ");
            builder.append(message);
        }
        return new ParseException(builder.toString());
    }

    /**
     * Returns the conversion of an option value to a type or a default value.
     * <p>
     * If the option is present, tries to convert its value to {@code <T>} using {@code converter}.<br>
     * If the conversion succeeds, returns its result, otherwise throws a {@link ParseException}.<br>
     * If the option is absent, returns the passed {@code def}.
     *
     * @param <T> The result type.
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value. <em>MAY BE</em> null.
     * @param converter Function used to convert a String to the expected type.
     * @return The conversion of the option value to {@code <Y>} type or {@code def}.
     * @throws ParseException If result could not be converted.
     */
    public static <T> T getValue(CommandLine cl,
                                 String opt,
                                 T def,
                                 Function<String, T> converter) throws ParseException {
        if (cl.hasOption(opt)) {
            try {
                return converter.apply(cl.getOptionValue(opt));
            } catch (final Exception e) {
                throw invalidArg(cl, opt, e.getMessage());
            }
        } else {
            return def;
        }
    }

    public static void fillValues(CommandLine cl,
                                  String opt,
                                  Collection<String> values) {
        if (cl.hasOption(opt)) {
            Collections.addAll(values, cl.getOptionValues(opt));
        }
    }

    public static <T> void fillValues(CommandLine cl,
                                      String opt,
                                      Collection<T> values,
                                      Function<String, T> converter) throws ParseException {
        Checks.isNotNull(values, "values");
        if (cl.hasOption(opt)) {
            try {
                for (final String s : cl.getOptionValues(opt)) {
                    values.add(converter.apply(s));
                }
            } catch (final Exception e) {
                throw invalidArg(cl, opt, e.getMessage());
            }
        }
    }

    public static <T> List<T> getValues(CommandLine cl,
                                        String opt,
                                        Function<String, T> converter) throws ParseException {
        final List<T> values = new ArrayList<>();
        fillValues(cl, opt, values, converter);
        return values;
    }

    public static List<String> getValues(CommandLine cl,
                                         String opt) {
        final List<String> values = new ArrayList<>();
        if (cl.hasOption(opt)) {
            Collections.addAll(values, cl.getOptionValues(opt));
        }
        return values;
    }

    /**
     * Returns an option value as a File.
     * <p>
     * If option is absent, returns a default value.<br>
     * Result is null only when option does not exist and default value is null.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a File.
     * @throws ParseException If result could not be produced.
     */
    public static File getValueAsFile(CommandLine cl,
                                      String opt,
                                      File def) throws ParseException {
        return getValue(cl, opt, def, File::new);
    }

    public static File getValueAsFile(CommandLine cl,
                                      String opt) throws ParseException {
        return getValueAsFile(cl, opt, null);
    }

    public static File getValueAsExistingFile(CommandLine cl,
                                              String opt,
                                              File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file == null || !file.isFile()) {
            throw invalidArg(cl, opt, "is not an existing file. Current dir: '" + Files.currentDir() + "'");
        }
        return file;
    }

    public static File getValueAsExistingFile(CommandLine cl,
                                              String opt) throws ParseException {
        return getValueAsExistingFile(cl, opt, null);
    }

    public static File getValueAsDirectory(CommandLine cl,
                                           String opt,
                                           File def) throws ParseException {
        final File result = getValueAsFile(cl, opt, def);
        if (result == null || (result.exists() && !result.isDirectory())) {
            throw invalidArg(cl, opt, "is not a directory. Current dir: '" + Files.currentDir() + "'");
        }
        if (!result.exists()) {
            final boolean done = result.mkdirs();
            if (!done) {
                throw new ParseException("Failed to create directory: " + result);
            }
        }
        return result;
    }

    public static File getValueAsDirectory(CommandLine cl,
                                           String opt) throws ParseException {
        return getValueAsDirectory(cl, opt, null);
    }

    /**
     * Returns an option value as an existing file or null.
     * <p>
     * If option is absent, returns a default value (possibly null).<br>
     * If result is not null and does not exist as a file, throws an exception.<br>
     * To obtain a null result :
     * <ul>
     * <li>default value must be set to null.</li>
     * <li>option must be absent.
     * </ul>
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value. <em>MAY BE</em> null.
     * @return The option value or default value. Is either an existing file or null.
     * @throws ParseException If result is not null and does not exist.
     */
    public static File getValueAsNullOrExistingFile(CommandLine cl,
                                                    String opt,
                                                    File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file != null && (!file.exists() || !file.isFile())) {
            throw invalidArg(cl, opt, "is not an existing file. Current dir: '" + Files.currentDir() + "'");
        }

        return file;
    }

    public static File getValueAsNullOrExistingFile(CommandLine cl,
                                                    String opt) throws ParseException {
        return getValueAsNullOrExistingFile(cl, opt, null);
    }

    public static File getValueAsExistingDirectory(CommandLine cl,
                                                   String opt,
                                                   File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file == null || !file.isDirectory()) {
            throw invalidArg(cl, opt, "is not an existing directory. Current dir: '" + Files.currentDir() + "'");
        }
        return file;
    }

    public static File getValueAsExistingDirectory(CommandLine cl,
                                                   String opt) throws ParseException {
        return getValueAsExistingDirectory(cl, opt, null);
    }

    /**
     * Returns an option value as an existing directory or null.
     * <p>
     * If option is absent, returns a default value (possibly null).<br>
     * If result is not null and does not exist as a directory, throws an exception.<br>
     * To obtain a null result :
     * <ul>
     * <li>default value must be set to null.</li>
     * <li>option must be absent.
     * </ul>
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value or default value. Is either an existing directory or null.
     * @throws ParseException If result is not null and does not exist.
     */
    public static File getValueAsNullOrExistingDirectory(CommandLine cl,
                                                         String opt,
                                                         File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file != null && (!file.exists() || !file.isDirectory())) {
            throw invalidArg(cl, opt, "is not an existing directory. Current dir: '" + Files.currentDir() + "'");
        }

        return file;
    }

    public static File getValueAsNullOrExistingDirectory(CommandLine cl,
                                                         String opt) throws ParseException {
        return getValueAsNullOrExistingDirectory(cl, opt, null);
    }

    public static File getValueAsExistingFileOrDirectory(CommandLine cl,
                                                         String opt,
                                                         File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file == null || !file.exists()) {
            throw invalidArg(cl, opt, "is not an existing file or directory. Current dir: '" + Files.currentDir() + "'");
        }
        return file;
    }

    public static File getValueAsExistingFileOrDirectory(CommandLine cl,
                                                         String opt) throws ParseException {
        return getValueAsExistingFileOrDirectory(cl, opt, null);
    }

    /**
     * Returns an option value as an existing file or directory or null.
     * <p>
     * If option is absent, returns a default value (possibly null).<br>
     * If result is not null and does not exist as a file or directory, throws an exception.<br>
     * To obtain a null result :
     * <ul>
     * <li>default value must be set to null.</li>
     * <li>option must be absent.
     * </ul>
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value or default value. Is either an existing file or directory or null.
     * @throws ParseException If result is not null and does not exist.
     */
    public static File getValueAsNullOrExistingFileOrDirectory(CommandLine cl,
                                                               String opt,
                                                               File def) throws ParseException {
        final File file = getValueAsFile(cl, opt, def);

        if (file != null && !file.exists()) {
            throw invalidArg(cl, opt, "is not an existing file or directory. Current dir: '" + Files.currentDir() + "'");
        }

        return file;
    }

    public static File getValueAsNullOrExistingFileOrDirectory(CommandLine cl,
                                                               String opt) throws ParseException {
        return getValueAsNullOrExistingFileOrDirectory(cl, opt, null);
    }

    /**
     * Returns an option value as an URL.
     * <p>
     * If option is absent, returns a default value (possibly null).<br>
     * If option is present and can be converted to an URL, returns this conversion.<br>
     * Otherwise, raises an exception.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value or default value.
     * @throws ParseException If option is present and can not be converted to a valid URL.
     */
    public static URL getValueAsURL(CommandLine cl,
                                    String opt,
                                    URL def) throws ParseException {
        if (cl.hasOption(opt)) {
            final String s = cl.getOptionValue(opt);
            try {
                return new URL(s);
            } catch (final MalformedURLException e) {
                throw new ParseException("Failed to convert '" + s + "' to URL (" + e.getMessage() + ")");
            }
        } else {
            return def;
        }
    }

    public static URL getValueAsURL(CommandLine cl,
                                    String opt) throws ParseException {
        return getValueAsURL(cl, opt, null);
    }

    /**
     * Returns an option value as a Charset.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a Charset.
     * @throws ParseException If the option value can not be parsed as a Charset.
     */
    public static Charset getValueAsCharset(CommandLine cl,
                                            String opt,
                                            Charset def) throws ParseException {
        if (cl.hasOption(opt)) {
            final String s = cl.getOptionValue(opt);
            try {
                return Charset.forName(s);
            } catch (final IllegalCharsetNameException | UnsupportedCharsetException e) {
                throw new ParseException("Failed to convert '" + s + "' to Charset (" + e.getMessage() + ")");
            }
        } else {
            return def;
        }
    }

    public static Charset getValueAsCharset(CommandLine cl,
                                            String opt) throws ParseException {
        return getValueAsCharset(cl, opt, Charset.defaultCharset());
    }

    /**
     * Returns an option value as a string.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a string.
     */
    public static String getValueAsString(CommandLine cl,
                                          String opt,
                                          String def) {
        return cl.getOptionValue(opt, def);
    }

    private static char parseChar(String s) {
        if (s != null && s.length() == 1) {
            return s.charAt(0);
        } else {
            throw new IllegalArgumentException("Invalid char '" + s + "'");
        }
    }

    /**
     * Returns an option value as a char.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a char.
     * @throws ParseException If the option value can not be parsed as a char.
     */
    public static char getValueAsChar(CommandLine cl,
                                      String opt,
                                      char def) throws ParseException {
        return getValue(cl, opt, def, AbstractMainSupport::parseChar);
    }

    public static Character getValueAsChar(CommandLine cl,
                                           String opt,
                                           Character def) throws ParseException {
        return getValue(cl, opt, def, AbstractMainSupport::parseChar);
    }

    /**
     * Returns an option value as a long.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a long.
     * @throws ParseException If the option value can not be parsed as a long.
     */
    public static long getValueAsLong(CommandLine cl,
                                      String opt,
                                      long def) throws ParseException {
        return getValue(cl, opt, def, Long::parseLong);
    }

    public static Long getValueAsLong(CommandLine cl,
                                      String opt,
                                      Long def) throws ParseException {
        return getValue(cl, opt, def, Long::parseLong);
    }

    /**
     * Returns an option value as an int.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as an int.
     * @throws ParseException If the option value can not be parsed as an int.
     */
    public static int getValueAsInt(CommandLine cl,
                                    String opt,
                                    int def) throws ParseException {
        return getValue(cl, opt, def, Integer::parseInt);
    }

    public static Integer getValueAsInt(CommandLine cl,
                                        String opt,
                                        Integer def) throws ParseException {
        return getValue(cl, opt, def, Integer::parseInt);
    }

    /**
     * Returns an option value as a short.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a short.
     * @throws ParseException If the option value can not be parsed as a short.
     */
    public static short getValueAsShort(CommandLine cl,
                                        String opt,
                                        short def) throws ParseException {
        return getValue(cl, opt, def, Short::parseShort);
    }

    public static Short getValueAsShort(CommandLine cl,
                                        String opt,
                                        Short def) throws ParseException {
        return getValue(cl, opt, def, Short::parseShort);
    }

    /**
     * Returns an option value as a byte.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a byte.
     * @throws ParseException If the option value can not be parsed as a byte.
     */
    public static byte getValueAsByte(CommandLine cl,
                                      String opt,
                                      byte def) throws ParseException {
        return getValue(cl, opt, def, Byte::parseByte);
    }

    public static Byte getValueAsByte(CommandLine cl,
                                      String opt,
                                      Byte def) throws ParseException {
        return getValue(cl, opt, def, Byte::parseByte);
    }

    /**
     * Returns an option value as a double.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a double.
     * @throws ParseException If the option value can not be parsed as a double.
     */
    public static double getValueAsDouble(CommandLine cl,
                                          String opt,
                                          double def) throws ParseException {
        return getValue(cl, opt, def, Double::parseDouble);
    }

    public static Double getValueAsDouble(CommandLine cl,
                                          String opt,
                                          Double def) throws ParseException {
        return getValue(cl, opt, def, Double::parseDouble);
    }

    /**
     * Returns an option value as a float.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param cl The command line.
     * @param opt The option name.
     * @param def Default value.
     * @return The option value as a float.
     * @throws ParseException If the option value can not be parsed as a float.
     */
    public static float getValueAsFloat(CommandLine cl,
                                        String opt,
                                        float def) throws ParseException {
        return getValue(cl, opt, def, Float::parseFloat);
    }

    public static Float getValueAsFloat(CommandLine cl,
                                        String opt,
                                        Float def) throws ParseException {
        return getValue(cl, opt, def, Float::parseFloat);
    }

    /**
     * Returns an option value as a enum.
     * <p>
     * If option is absent, returns a default value.
     *
     * @param <E> The enum type.
     * @param cl The command line.
     * @param opt The option name.
     * @param enumClass Enum class
     * @param def Default value.
     * @return The option value as an enum.
     * @throws ParseException If the option value can not be parsed as an enum value.
     */
    public static <E extends Enum<E>> E getValueAsEnum(CommandLine cl,
                                                       String opt,
                                                       Class<E> enumClass,
                                                       E def) throws ParseException {
        return getValue(cl, opt, def, s -> {
            for (final E e : enumClass.getEnumConstants()) {
                if (e.name().equals(s)) {
                    return e;
                }
            }
            throw new IllegalArgumentException("Invalid enum value");
        });
    }

    public static int getNumberOfParts(String s,
                                       String sep) {
        int count = 0;
        int from = 0;
        while (from >= 0) {
            final int pos = s.indexOf(sep, from);
            if (pos >= 0) {
                count++;
                from = pos + sep.length();
            } else {
                from = -1;
            }
        }
        return count + 1;
    }

    public static String getPart(String s,
                                 String sep,
                                 int index,
                                 String def) {
        int count = 0;
        int from = 0;
        while (from >= 0) {
            final int pos = s.indexOf(sep, from);
            if (pos >= 0) {
                if (count == index) {
                    return s.substring(from, pos);
                }
                count++;
                from = pos + sep.length();
            } else {
                if (count == index) {
                    return s.substring(from);
                }
                from = -1;
            }
        }
        return def;
    }

    public static String getPart(String s,
                                 String sep,
                                 int index) {
        return getPart(s, sep, index, null);
    }
}