package cdc.util.cli;

import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import cdc.util.lang.Checks;

/**
 * Management of extensible boolean options.
 */
public class OptionEnumType {
    private final Map<String, Value> values = new HashMap<>();

    private Value add(Value value) {
        Checks.doesNotContainKey(values, value.getName(), "values");
        values.put(value.getName(), value);
        return value;
    }

    private record Value(String name,
                         String shortName,
                         String description)
            implements OptionEnum {
        public Value {
            Checks.isNotNull(name, "name");
            Checks.isNotNull(description, "description");
        }

        @Override
        public String getName() {
            return name;
        }

        @Override
        public String getShortName() {
            return shortName;
        }

        @Override
        public String getDescription() {
            return description;
        }
    }

    public Collection<? extends OptionEnum> getValues() {
        return values.values();
    }

    public boolean contains(OptionEnum value) {
        return values.containsKey(value.getName())
                && values.get(value.getName()) == value;
    }

    public Builder value() {
        return new Builder(this);
    }

    public Builder value(String name) {
        return value().name(name);
    }

    public Builder value(String name,
                         String description) {
        return value().name(name).description(description);
    }

    public static class Builder {
        private final OptionEnumType owner;
        private String name;
        private String shortName = null;
        private String description = "TODO";

        private Builder(OptionEnumType owner) {
            this.owner = owner;
        }

        public Builder init(OptionEnum other) {
            this.name = other.getName();
            this.shortName = other.getShortName();
            this.description = other.getDescription();
            return this;
        }

        public Builder name(String name) {
            this.name = name;
            return this;
        }

        public Builder shortName(String shortName) {
            this.shortName = shortName;
            return this;
        }

        public Builder description(String description) {
            this.description = description;
            return this;
        }

        public OptionEnum build() {
            final Value value = new Value(name, shortName, description);
            return owner.add(value);
        }
    }
}