/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * <p>
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * <p>
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * <p>
 * 3. Neither the name of Liferay, Inc. nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package com.wedeploy.api.sdk;

import com.wedeploy.api.WeDeploy;
import com.wedeploy.api.WeDeployException;

import java.util.Collection;

/**
 * Auth.
 */
public interface Auth {

	/**
	 * Creates authentication instance.
	 */
	public static Auth create(String token) {
		return new AuthImpl(token);
	}

	/**
	 * Creates authentication instance.
	 */
	public static Auth create(String email, String password) {
		return new AuthImpl(email, password);
	}

	/**
	 * Creates master authentication instance.
	 */
	public static Auth master() {
		if (WeDeploy.MASTER_TOKEN == null) {
			throw new WeDeployException(
				"Master token is not available. To use this method, " +
					"set a value for WeDeploy.MASTER_TOKEN.");
		}

		return new AuthImpl(WeDeploy.MASTER_TOKEN);
	}

	/**
	 * Returns user email.
	 */
	public String email();

	/**
	 * Returns {@code true} if {@link #email()} is defined.
	 */
	public default boolean hasEmail() {
		String email = email();

		if (email != null && !email.isEmpty()) {
			return true;
		}

		return false;
	}

	/**
	 * Returns {@code true} if {@link #id()} is defined.
	 */
	public default boolean hasId() {
		String id = id();

		if (id != null && !id.isEmpty()) {
			return true;
		}

		return false;
	}

	/**
	 * Returns {@code true} if {@link #name()} is defined.
	 */
	public default boolean hasName() {
		String name = name();

		if (name != null && !name.isEmpty()) {
			return true;
		}

		return false;
	}

	/**
	 * Returns {@code true} if {@link #password()} is defined.
	 */
	public default boolean hasPassword() {
		String password = password();

		if (password != null && !password.isEmpty()) {
			return true;
		}

		return false;
	}

	/**
	 * Returns {@code true} if {@link #photoUrl()} is defined.
	 */
	public default boolean hasPhotoUrl() {
		String photoUrl = photoUrl();

		if (photoUrl != null && !photoUrl.isEmpty()) {
			return true;
		}

		return false;
	}

	/**
	 * Returns {@code true} if all of the scopes are defined.
	 */
	public default boolean hasScope(Collection<String> scopes) {
		return scopes
			.stream()
			.map(this::hasScope)
			.reduce(Boolean.TRUE, Boolean::logicalAnd);
	}

	/**
	 * Returns {@code true} if provided scope is defined.
	 */
	public boolean hasScope(String scope);

	/**
	 * Returns {@code true} if {@link #token()} is defined.
	 */
	public default boolean hasToken() {
		String token = token();

		if (token != null && !token.isEmpty()) {
			return true;
		}

		return false;
	}

	/**
	 * Returns user id.
	 */
	public String id();

	/**
	 * Is the user authenticated.
	 */
	public boolean isAuthenticated();

	/**
	 * Checks if the user is master user.
	 */
	public default boolean isMaster() {
		if (WeDeploy.MASTER_TOKEN != null &&
			WeDeploy.MASTER_TOKEN.equals(token())) {

			return true;
		}

		return false;
	}

	/**
	 * Returns a user name.
	 */
	public String name();

	/**
	 * Returns password.
	 */
	public String password();

	/**
	 * Returns photo URL.
	 */
	public String photoUrl();

	/**
	 * Returns authentication token.
	 */
	public String token();

}
