/**
 * Copyright (c) 2000-present Liferay, Inc. All rights reserved.
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * <p>
 * 1. Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * <p>
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * <p>
 * 3. Neither the name of Liferay, Inc. nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * <p>
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

package com.wedeploy.api.util;

import com.wedeploy.api.sdk.MultiMap;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

public class Util {

	public static final String SEPARATOR = "/";

	/**
	 * Adds parameters into the url querystring.
	 */
	public static String addParametersToQueryString(
		String query, MultiMap<String> params) {

		StringBuilder queryString = new StringBuilder();

		if ((query != null) && (query.length() > 0)) {
			queryString.append(query).append('&');
		}

		params.names().forEach(
			name -> params.getAll(name).forEach(value -> {
				queryString.append(name);
				queryString.append("=");
				queryString.append(encodeURIComponent(value));
				queryString.append("&");
			}));

		if (params.size() > 0) {
			queryString.deleteCharAt(queryString.length() - 1);
		}

		return queryString.toString();
	}

	/**
	 * Simulates Javascript encodeURIComponent.
	 */
	public static String encodeURIComponent(String s) {
		try {
			return URLEncoder.encode(s, DEFAULT_ENCODING)
				.replaceAll("\\+", "%20")
				.replaceAll("\\%21", "!")
				.replaceAll("\\%27", "'")
				.replaceAll("\\%28", "(")
				.replaceAll("\\%29", ")")
				.replaceAll("\\%7E", "~");
		} catch (UnsupportedEncodingException e) {
			return s;
		}
	}

	/**
	 * Parses the url separating the domain and port from the path.
	 */
	public static String joinPathAndQuery(String path, String query) {
		StringBuilder builder = new StringBuilder();

		if (path != null) {
			builder.append(path);
		}

		if ((query != null) && (query.length() > 0)) {
			builder.append('?').append(query);
		}

		return builder.toString();
	}

	/**
	 * Joins two paths.
	 */
	public static String joinPaths(String basePath, String path) {
		if (basePath.endsWith(SEPARATOR)) {
			basePath = basePath.substring(0, basePath.length() - 1);
		}

		if (path.startsWith(SEPARATOR)) {
			path = path.substring(1);
		}

		return (basePath + SEPARATOR + path).replaceAll("\\/$", "");
	}

	protected Util() {
	}

	protected static String DEFAULT_ENCODING = "UTF-8";

}
