package dev.goquick.sqlitenow.gradle.context

import dev.goquick.sqlitenow.gradle.model.AnnotatedCreateTableStatement
import dev.goquick.sqlitenow.gradle.model.AnnotatedCreateViewStatement
import dev.goquick.sqlitenow.gradle.model.AnnotatedExecuteStatement
import dev.goquick.sqlitenow.gradle.model.AnnotatedSelectStatement
import dev.goquick.sqlitenow.gradle.model.AnnotatedStatement
import dev.goquick.sqlitenow.gradle.sqlinspect.DeleteStatement
import dev.goquick.sqlitenow.gradle.sqlinspect.InsertStatement
import dev.goquick.sqlitenow.gradle.sqlinspect.SelectStatement
import dev.goquick.sqlitenow.gradle.sqlinspect.UpdateStatement

/**
 * Centralized logic for finding columns associated with parameters
 * across different statement types (SELECT, INSERT, DELETE).
 */
class ColumnLookup(
    createTableStatements: List<AnnotatedCreateTableStatement>,
    createViewStatements: List<AnnotatedCreateViewStatement>,
) {

    private val tableLookup: Map<String, AnnotatedCreateTableStatement> =
        createTableStatements.associateBy { it.src.tableName.lowercase() }
    private val viewLookup: Map<String, AnnotatedCreateViewStatement> =
        createViewStatements.associateBy { it.src.viewName.lowercase() }

    fun findTableByName(tableName: String): AnnotatedCreateTableStatement? {
        return tableLookup[tableName.lowercase()]
    }

    fun findViewByName(viewName: String): AnnotatedCreateViewStatement? {
        return viewLookup[viewName.lowercase()]
    }

    fun findColumnForParameter(
        statement: AnnotatedStatement,
        paramName: String
    ): AnnotatedCreateTableStatement.Column? {
        return when (statement) {
            is AnnotatedSelectStatement -> findColumnForSelectParameter(statement, paramName)
            is AnnotatedExecuteStatement -> findColumnForExecuteParameter(statement, paramName)
            else -> null
        }
    }

    /**
     * Helper function to find the column for a SELECT statement parameter.
     */
    private fun findColumnForSelectParameter(
        statement: AnnotatedSelectStatement,
        paramName: String
    ): AnnotatedCreateTableStatement.Column? {
        val tableName = statement.src.fromTable ?: return null

        // First try to find as a table
        val table = findTableByName(tableName)
        if (table != null) {
            val associatedColumn = statement.src.namedParametersToColumns[paramName]
            if (associatedColumn != null) {
                return table.findColumnByName(associatedColumn.columnName)
            }

            val paramLower = paramName.lowercase()
            return table.findColumnByName(paramName) ?: table.columns.find { col ->
                statement.annotations.propertyNameGenerator.convertToPropertyName(col.src.name)
                    .lowercase() == paramLower
            }
        }

        // If not found as a table, try to find as a view
        val view = findViewByName(tableName)
        if (view != null) {
            return findColumnForViewParameter(view, statement, paramName)
        }

        return null
    }

    /**
     * Find the column for an EXECUTE statement (INSERT/DELETE/UPDATE) parameter.
     * Returns the column if found, null otherwise.
     */
    private fun findColumnForExecuteParameter(
        statement: AnnotatedExecuteStatement,
        paramName: String
    ): AnnotatedCreateTableStatement.Column? {
        val table = findTableByName(statement.src.table) ?: return null

        fun findColumnInWithClause(stmts: List<SelectStatement>): AnnotatedCreateTableStatement.Column? {
            for (withSelectStatement in stmts) {
                if (paramName in withSelectStatement.namedParameters) {
                    val associatedColumn = withSelectStatement.namedParametersToColumns[paramName]
                    if (associatedColumn != null) {
                        val fromTable = withSelectStatement.fromTable
                        if (fromTable != null) {
                            val withTable = findTableByName(fromTable) ?: continue
                            return withTable.findColumnByName(associatedColumn.columnName)
                        }
                    }
                }
            }
            return null
        }

        return when (val src = statement.src) {
            is InsertStatement -> {
                val columnName = src.columnNamesAssociatedWithNamedParameters[paramName]
                columnName?.let(table::findColumnByName)
                    ?: findColumnInWithClause(src.withSelectStatements)
            }

            is DeleteStatement -> {
                val associatedColumn = src.namedParametersToColumns[paramName]
                associatedColumn?.let { table.findColumnByName(it.columnName) }
                    ?: findColumnInWithClause(src.withSelectStatements)
            }

            is UpdateStatement -> {
                // First check if this parameter is from a SET clause (has direct column mapping)
                val directColumnName = src.namedParametersToColumnNames[paramName]
                directColumnName?.let(table::findColumnByName)
                    ?: run {
                        // If not from SET clause, check WHERE clause parameters
                        val associatedColumn = src.namedParametersToColumns[paramName]
                        associatedColumn?.let { table.findColumnByName(it.columnName) }
                            ?: findColumnInWithClause(src.withSelectStatements)
                    }
            }
        }
    }

    /** Find a column in a table using both direct name match and property name conversion. */
    private fun findColumnInTable(
        table: AnnotatedCreateTableStatement,
        columnName: String,
        paramName: String,
        statement: AnnotatedSelectStatement
    ): AnnotatedCreateTableStatement.Column? {
        // First try direct column name match
        val column = table.findColumnByName(columnName)
        if (column != null) {
            return column
        }

        // Then try property name conversion
        val paramLower = paramName.lowercase()
        return table.columns.find { col ->
            statement.annotations.propertyNameGenerator.convertToPropertyName(col.src.name)
                .lowercase() == paramLower
        }
    }

    /**
     * Find the column for a VIEW-based SELECT statement parameter.
     * Looks up the column in the underlying tables that the VIEW references and merges
     * any field annotations from the VIEW definition.
     */
    private fun findColumnForViewParameter(
        view: AnnotatedCreateViewStatement,
        statement: AnnotatedSelectStatement,
        paramName: String
    ): AnnotatedCreateTableStatement.Column? {
        val associatedColumn = statement.src.namedParametersToColumns[paramName]
        val columnName = associatedColumn?.columnName ?: paramName

        resolveColumnFromView(
            view = view,
            fieldAlias = columnName,
            paramName = paramName,
            statement = statement,
            visitedViews = mutableSetOf()
        )?.let { return it }

        return null
    }

    private fun resolveColumnFromView(
        view: AnnotatedCreateViewStatement,
        fieldAlias: String,
        paramName: String,
        statement: AnnotatedSelectStatement,
        visitedViews: MutableSet<String>,
    ): AnnotatedCreateTableStatement.Column? {
        val viewKey = view.src.viewName.lowercase()
        if (!visitedViews.add(viewKey)) {
            return null
        }

        val candidates = view.fields.filter { field ->
            field.src.fieldName.equals(fieldAlias, ignoreCase = true) ||
                field.src.originalColumnName.equals(fieldAlias, ignoreCase = true)
        }

        candidates.forEach { field ->
            val sourceTableName = field.src.tableName
            val originalColumnName = field.src.originalColumnName.ifBlank { field.src.fieldName }

            if (sourceTableName.isNotBlank()) {
                findTableByName(sourceTableName)?.let { table ->
                    findColumnInTable(table, originalColumnName, paramName, statement)?.let { return it }
                }

                findViewByName(sourceTableName)?.let { nestedView ->
                    resolveColumnFromView(
                        view = nestedView,
                        fieldAlias = originalColumnName,
                        paramName = paramName,
                        statement = statement,
                        visitedViews = visitedViews
                    )?.let { return it }
                }
            }
        }

        val selectStatement = view.src.selectStatement

        selectStatement.fromTable?.let { fromTableName ->
            findTableByName(fromTableName)?.let { table ->
                findColumnInTable(table, fieldAlias, paramName, statement)?.let { return it }
            }

            findViewByName(fromTableName)?.let { nestedView ->
                resolveColumnFromView(
                    view = nestedView,
                    fieldAlias = fieldAlias,
                    paramName = paramName,
                    statement = statement,
                    visitedViews = visitedViews
                )?.let { return it }
            }
        }

        for (joinTableName in selectStatement.joinTables) {
            findTableByName(joinTableName)?.let { table ->
                findColumnInTable(table, fieldAlias, paramName, statement)?.let { return it }
            }

            findViewByName(joinTableName)?.let { nestedView ->
                resolveColumnFromView(
                    view = nestedView,
                    fieldAlias = fieldAlias,
                    paramName = paramName,
                    statement = statement,
                    visitedViews = visitedViews
                )?.let { return it }
            }
        }

        return null
    }


    /** Checks if a parameter's corresponding column is nullable. */
    fun isParameterNullable(statement: AnnotatedStatement, paramName: String): Boolean {
        val column = findColumnForParameter(statement, paramName) ?: return false
        return column.isNullable()
    }
}
