package fi.evolver.script.app;

import java.io.IOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.time.Duration;
import java.util.Base64;
import java.util.Optional;
import java.util.regex.Pattern;

import fi.evolver.script.Dialog;
import fi.evolver.script.FileUtils;
import fi.evolver.script.Shell;
import fi.evolver.script.Step;

public class Maven {
	private static final Dialog.DialogTextField DIALOG_FIELD_USERNAME = Dialog.textField("username");
	private static final Dialog.DialogTextField DIALOG_FIELD_PASSWORD = Dialog.passwordField("password");

	private static final Duration VALIDATION_TIMEOUT = Duration.ofSeconds(10);

	public static void writeSettings(Path source) throws IOException {
		try (Step step = Step.start("Maven: write settings")) {
			Path mavenSettingsPath = Shell.HOME.resolve(".m2/settings.xml");
			Files.createDirectories(mavenSettingsPath.getParent());
			Files.copy(source, mavenSettingsPath, StandardCopyOption.REPLACE_EXISTING);
		}
	}

	public static void setupMavenCredentials(Repo repo) {
		try (Step step = Step.start("Setup Maven credentials")) {
			Path credentialsFile = Shell.HOME.resolve(".maven-credentials-%s".formatted(repo.name.toLowerCase()));

			if (hasValidCredentials(repo, credentialsFile)) {
				step.skip("Already valid");
				return;
			}

			Credentials credentials = readAndValidateCredentials(repo);

			FileUtils.write(credentialsFile, """
					export %s_MAVEN_USERNAME="%s"
					export %s_MAVEN_PASSWORD="%s"
					"""
					.formatted(
							repo.name.toUpperCase(),
							credentials.username,
							repo.name.toUpperCase(),
							credentials.password
					)
			);

			FileUtils.writeShellBlock(Shell.BASHRC, "MavenCredentials-%s".formatted(repo.name), "source %s".formatted(credentialsFile.toString()));
		}
	}

	private static Credentials readAndValidateCredentials(Repo repo) {
		Optional<String> errorInfo = Optional.empty();

		while (true) {
			Dialog dialog = new Dialog("Maven credentials: %s".formatted(repo.name))
					.add(DIALOG_FIELD_USERNAME)
					.add(DIALOG_FIELD_PASSWORD);

			errorInfo
					.map("Error: %s"::formatted)
					.map(Dialog::paragraph)
					.ifPresent(dialog::add);

			var dialogResult = dialog.show();

			Credentials credentials = new Credentials(
					dialogResult.get(DIALOG_FIELD_USERNAME),
					dialogResult.get(DIALOG_FIELD_PASSWORD)
			);

			try {
				var validationResponse = validateCredentials(credentials, repo.pingUrl);
				if (validationResponse.statusCode() == 200)
					return credentials;

				errorInfo = Optional.of("%s %s".formatted(validationResponse.statusCode(), validationResponse.body()));
			} catch (IOException e) {
				errorInfo = Optional.of(e.toString());
			}
		}
	}

	private static HttpResponse<String> validateCredentials(Credentials credentials, URI repoPingUrl) throws IOException {
		try (HttpClient client = HttpClient.newBuilder().connectTimeout(VALIDATION_TIMEOUT).build()) {

			String authHeader = "Basic " + Base64.getEncoder().encodeToString(
					(credentials.username + ":" + credentials.password).getBytes());

			HttpRequest request = HttpRequest.newBuilder()
					.uri(repoPingUrl)
					.header("Authorization", authHeader)
					.timeout(VALIDATION_TIMEOUT)
					.GET()
					.build();

			return client.send(request, HttpResponse.BodyHandlers.ofString());
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
			throw new RuntimeException("Credential validation interrupted", e);
		}
	}

	private static boolean hasValidCredentials(Repo repo, Path credentialsFile) {
		Optional<Credentials> existingCredentials = readExistingCredentials(repo, credentialsFile);
		if (existingCredentials.isEmpty())
			return false;

		try {
			return validateCredentials(existingCredentials.get(), repo.pingUrl).statusCode() == 200;
		} catch (IOException ignore) {
			return false;
		}
	}

	private static Optional<Credentials> readExistingCredentials(Repo repo, Path credentialsFile) {
		try {
			if (!Files.exists(credentialsFile))
				return Optional.empty();

			String content = Files.readString(credentialsFile);

			var usernameMatcher = Pattern
					.compile("%s_MAVEN_USERNAME=\"?([^\"]*)\"?".formatted(repo.name.toUpperCase()))
					.matcher(content);

			if (!usernameMatcher.find())
				return Optional.empty();

			var passwordMatcher = Pattern
					.compile("%s_MAVEN_PASSWORD=\"?([^\"]*)\"?".formatted(repo.name.toUpperCase()))
					.matcher(content);

			if (!passwordMatcher.find())
				return Optional.empty();

			return Optional.of(new Credentials(
					usernameMatcher.group(1),
					passwordMatcher.group(1)));

		} catch (IOException e) {
			return Optional.empty();
		}
	}

	public static Repo repo(String name, URI pingUrl) {
		return new Repo(name, pingUrl);
	}

	public record Repo(
			String name,
			URI pingUrl
	) {
	}

	private record Credentials(
			String username,
			String password
	) {
	}
}
