package fi.evolver.script.app;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import fi.evolver.script.FileUtils;
import fi.evolver.script.Shell;
import fi.evolver.script.Shell.Command;
import fi.evolver.script.Shell.Result;
import fi.evolver.script.Step;

public class Nvm {
	private static final String REPOSITORY_URL = "https://github.com/nvm-sh/nvm.git";

	private static final String CONFIG = """
			export NVM_DIR="$HOME/.nvm"
			[ -s "$NVM_DIR/nvm.sh" ] && \\. "$NVM_DIR/nvm.sh"  # This loads nvm
			[ -s "$NVM_DIR/bash_completion" ] && \\. "$NVM_DIR/bash_completion"
			""";

	public static Node use(Path workingDirectory) {
		return use(workingDirectory, null);
	}

	public static Node use(String version) {
		return use(Path.of("."), version);
	}

	public static Node use(Path workingDirectory, String versionOverride) {
		try (Step step = Step.start("Nvm: use version %s".formatted(versionOverride))) {
			if (workingDirectory == null || !Files.isDirectory(workingDirectory))
				step.fail("Invalid working directory '%s'".formatted(workingDirectory));

			install();

			String version = versionOverride != null
					? versionOverride
					: getVersionFromNvmRc(workingDirectory);

			String exactVersion = Nvm.command("version", version).stdout().trim();
			if ("N/A".equalsIgnoreCase(exactVersion)) {
				if (!Nvm.command("install", version).success())
					step.fail("Could not find npm version %s".formatted(version));
				exactVersion = Nvm.command("version", version).stdout().trim();
			}
			Path result = Shell.HOME.resolve(".nvm/versions/node").resolve(exactVersion).resolve("bin");
			if (!Files.isDirectory(result))
				step.fail("Could not find the npm directory for version %s (%s)".formatted(exactVersion, result));
			return new Node(result, workingDirectory);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	private static String getVersionFromNvmRc(Path workingDirectory) throws IOException {
		Path nvmRc = resolveNvmrc(workingDirectory).orElseThrow();
		return Files.readString(nvmRc).trim();
	}

	private static Optional<Path> resolveNvmrc(Path workingDirectory) {
		while (workingDirectory != null) {
			if (Files.exists(workingDirectory.resolve(".nvmrc"))) {
				return Optional.of(workingDirectory.resolve(".nvmrc"));
			}
			workingDirectory = workingDirectory.getParent();
		}
		return Optional.empty();
	}


	public static void install() {
		try (Step step = Step.start("Nvm: install")) {
			Path targetDir = Shell.HOME.resolve(".nvm");
			if (Files.exists(targetDir)) {
				step.skip("Already installed");
				return;
			}

			Git.clone(REPOSITORY_URL, targetDir);

			try {
				FileUtils.writeShellBlock(Shell.BASHRC, "NVM", CONFIG);
			}
			catch (RuntimeException e) {
				step.fail("Could not write configuration", e);
			}
		}
	}


	private static Result command(String... args) {
		String escaped = Arrays.stream(args).map("'%s'"::formatted).collect(Collectors.joining(" "));
		try (Step step = Step.start("npm %s".formatted(escaped))) {
			if (Arrays.stream(args).anyMatch(c -> c.contains("'")))
				throw new IllegalArgumentException("Nvm command contains ', which is not supported");

			return Command.user("bash", "-c", "export NVM_DIR=$HOME/.nvm; source $NVM_DIR/nvm.sh; nvm %s".formatted(escaped))
					.failOnError(false)
					.run();
		}
	}


	public record Node(Path bin, Path workingDirectory) {

        public Command command(String... args) {
            return command(Arrays.asList(args));
        }

        public Command command(List<String> args) {
            List<String> fullCommand = new ArrayList<>();
            fullCommand.add(bin().resolve("node").toString());
            fullCommand.addAll(args);
            return shellCommand(fullCommand);
        }

        public Command shellCommand(String... command) {
            return shellCommand(Arrays.asList(command));
        }

        public Command shellCommand(List<String> command) {
            try (Step step = Step.start("Node: %s".formatted(command))) {
                return Command
                        .user(command)
                        .workingDirectory(workingDirectory)
                        .env("PATH", "%s:%s".formatted(bin(), System.getenv("PATH")));
            }
        }


        public void npm(String... args) {
            npm(Arrays.asList(args));
        }

        public void npm(List<String> args) {
            try (Step step = Step.start("Npm: %s".formatted(String.join(" ", args)))) {
                List<String> fullCommand = new ArrayList<>(args.size() + 1);
                fullCommand.add(bin().resolve("npm").toString());
                fullCommand.addAll(args);
                command(fullCommand).run();
            }
        }

	}

}
