package io.bitsensor.plugins.java.sql;

import io.bitsensor.lib.entity.proto.Invocation.SQLInvocation.Builder;
import io.bitsensor.lib.entity.proto.Invocation.SQLInvocation.Query;
import io.bitsensor.plugins.java.core.BitSensor;
import io.bitsensor.plugins.java.core.BitSensorDI;
import io.bitsensor.plugins.java.sql.handler.SQLStatementHandlerManager;
import org.apache.tomcat.jdbc.pool.interceptor.AbstractCreateStatementInterceptor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.*;
import java.sql.*;
import java.util.*;

import static io.bitsensor.lib.entity.proto.Invocation.SQLInvocation.newBuilder;

/**
 * Abstract JDBC Interceptor class that implements interceptor for SQL executions and logs them to BitSensor.
 */
public abstract class BaseJdbcInterceptor extends AbstractCreateStatementInterceptor {

    private static final Map<String, Constructor<?>> CONSTRUCTORS = new HashMap<>();
    private static final Logger LOGGER = LoggerFactory.getLogger(BaseJdbcInterceptor.class);

    /**
     * List of queries to be executed in batch.
     */
    private static final List<Query> BATCH_QUERIES = new ArrayList<>();

    private SQLStatementHandlerManager sqlStatementHandlerManager;

    public BaseJdbcInterceptor() {
        sqlStatementHandlerManager = BitSensorDI.getBean(SQLStatementHandlerManager.class);
    }

    /**
     * Creates a constructor for a proxy class, if one doesn't already exist.
     *
     * @param clazz the interface that the proxy will implement.
     * @return a constructor used to create new instances.
     * @throws NoSuchMethodException if constructor method of the proxy class is not found.
     */
    private Constructor<?> getConstructor(Class<?> clazz) throws NoSuchMethodException {
        if (!CONSTRUCTORS.containsKey(clazz.getName())) {
            Class<?> proxyClass = Proxy.getProxyClass(BaseJdbcInterceptor.class.getClassLoader(), clazz);
            CONSTRUCTORS.put(clazz.getName(), proxyClass.getConstructor(InvocationHandler.class));
        }

        return CONSTRUCTORS.get(clazz.getName());
    }

    @Override
    public Object createStatement(Object proxy, Method method, Object[] args, Object statement, long time) {
        try {

            String name = method.getName();
            String sql = null;
            Constructor<?> constructor;

            Builder sqlInv = getSQLInvocationBuilder(statement);

            if (compare(CREATE_STATEMENT, name)) {
                // createStatement
                constructor = getConstructor(Statement.class);
            } else if (compare(PREPARE_STATEMENT, name)) {
                // prepareStatement
                constructor = getConstructor(PreparedStatement.class);
                sql = (String) args[0];
                sqlInv.setPrepareStatement(sql);
            } else if (compare(PREPARE_CALL, name)) {
                // prepareCall
                constructor = this.getConstructor(CallableStatement.class);
                sql = (String) args[0];
                sqlInv.setPrepareCall(sql);
            } else {
                return statement;
            }

            return constructor.newInstance(new BaseJdbcInterceptor.StatementProxy(statement, sql, sqlInv));
        } catch (IllegalAccessException | InstantiationException | InvocationTargetException | NoSuchMethodException | SQLException e) {
            LOGGER.warn("Unable to create statement proxy for BitSensor.", e);
        }
        return statement;
    }

    private Builder getSQLInvocationBuilder(Object statement) throws SQLException {
        Statement statement0 = (Statement) statement;
        Connection connection = statement0.getConnection();
        DatabaseMetaData metaData = connection.getMetaData();

        Builder sqlInv = newBuilder();
        sqlInv.putEndpoint("url", metaData.getURL());
        sqlInv.putEndpoint("catalog", connection.getCatalog());
        sqlInv.putEndpoint("user", metaData.getUserName());
        sqlInv.putEndpoint("driver_version", metaData.getDriverVersion());
        sqlInv.putEndpoint("database_type", metaData.getDatabaseProductName());
        sqlInv.putEndpoint("database_version", metaData.getDatabaseProductVersion());

        return sqlInv;
    }

    @Override
    public void closeInvoked() {
        BATCH_QUERIES.clear();
    }

    /**
     * Returns formatted query string.
     *
     * @param query query string
     * @return formatted query string
     */
    protected abstract String queryFormat(String query);

    private class StatementProxy implements InvocationHandler {
        private boolean closed = false;
        private Object delegate;
        private String query;
        private Builder builder;

        StatementProxy(Object parent, String query, Builder builder) {
            this.delegate = parent;
            this.query = query;
            this.builder = builder;
        }


        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {

            // allow close() to be called multiple times
            boolean close = compare(CLOSE_VAL, method);
            if (close && closed)
                return null;
            // if calling isClosed()
            if (compare(ISCLOSED_VAL, method))
                return closed;
            // if calling something else
            if (closed)
                throw new SQLException("Statement closed.");

            // checks to see if we are about to execute a query
            boolean process = isExecute(method, false);

            Object result;

            // execution timer
            long start = 0;

            if (compare("addBatch", method)) {
                BATCH_QUERIES.add(Query.newBuilder().setQuery(queryFormat(delegate.toString())).build());
            }

            // pre handle
            if (process) {
                // saves query
                if (query == null && args.length == 1) {
                    builder.addQueries(Query.newBuilder().setQuery(queryFormat(String.valueOf(args[0]))));
                } else if (compare(EXECUTE_BATCH, method)) {
                    // when execute batch
                    builder.addAllQueries(BATCH_QUERIES);
                    BATCH_QUERIES.clear();
                } else {
                    // when execute with PreparedStatement
                    builder.addQueries(Query.newBuilder().setQuery(queryFormat(delegate.toString())));
                }

                start = System.currentTimeMillis();
                builder
                        .putEndpoint("start_time", String.valueOf(start))
                        .putEndpoint("successful", String.valueOf(false));
                sqlStatementHandlerManager.preHandle((Statement) this.delegate, builder);
            }

            try {
                // invokes
                result = method.invoke(this.delegate, args);
            } catch (Throwable t) {
                Throwable throwable = (t instanceof InvocationTargetException && t.getCause() != null)
                        ? t.getCause()
                        : t;
                BitSensor.addThrowable(throwable);
                throw throwable;
            }

            // post handle
            if (process) {
                long delta = System.currentTimeMillis() - start;
                builder
                        .putEndpoint("successful", String.valueOf(true))
                        .putEndpoint("result", (result instanceof int[])
                                ? Arrays.toString((int[]) result)
                                : String.valueOf(result))
                        .putEndpoint("execution_time", String.valueOf(delta));
                sqlStatementHandlerManager.postHandle((Statement) this.delegate, builder);
            }

            if (close) {
                this.closed = true;
                this.delegate = null;
            }
            return result;
        }
    }
}
