package amk.sdk.deeplink.presenter;

import amk.sdk.deeplink.entity.model.*;
import amk.sdk.deeplink.exception.DeeplinkException;
import amk.sdk.deeplink.entity.request.IndividualRequest;
import amk.sdk.deeplink.entity.request.MerchantRequest;
import amk.sdk.deeplink.entity.response.AMKDLResponse;
import com.google.gson.Gson;
import okhttp3.*;
import org.jetbrains.annotations.NotNull;

import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

import static amk.sdk.deeplink.entity.response.DeeplinkResponseMapper.convertToDeeplinkResponseBody;
import static amk.sdk.deeplink.entity.response.PaymentStatusResponseMapper.convertToPaymentStatusResponseBody;
import static amk.sdk.deeplink.exception.DeeplinkException.throwCustomerException;
import static amk.sdk.deeplink.entity.model.ApiConstant.BASE_URL;

public class DeeplinkPresenter {
    private static String mClientId;
    private static String mSecretKey;
    private static String mJwtAudience;
    private static OkHttpClient client = new OkHttpClient().newBuilder().build();
    private static MediaType mediaType = MediaType.parse("application/json");
    private static Gson gson = new Gson();

    public static void preConfig(String clientId, String secretKey, String jwtAudience) {
        mClientId = clientId;
        mSecretKey = secretKey;
        mJwtAudience = jwtAudience;
    }

    public static AMKDLResponse<AMKDeeplinkData> generateIndividualDeeplink(AMKDLIndividual amkdlIndividual) throws DeeplinkException {
        return okHttpClientGenerator(amkdlIndividual, true);

    }

    public static AMKDLResponse<AMKDeeplinkData> generateMerchantDeeplink(AMKDLMerchant amkdlMerchant) throws DeeplinkException {
        return okHttpClientGenerator(amkdlMerchant, false);

    }

    public static AMKDLResponse<AMKPaymentStatusData> checkPaymentStatus(String referenceId) throws DeeplinkException {
        String endpoint = "/deep-link/payment/status";
        String accessToken = getAccessToken();

        String requestBody = gson.toJson(AMKRequestBody.getReferenceIdRequest(referenceId));
        RequestBody body = RequestBody.create(mediaType, requestBody);
        Request request = new Request.Builder()
                .url(BASE_URL + endpoint)
                .method("POST", body)
                .addHeader("Authorization", "Bearer " + accessToken)
                .build();
        try {
            Response response = client.newCall(request).execute();
            return convertToPaymentStatusResponseBody(response);
        } catch (Exception e) {
            exceptionMapper(e);
        }
        return null;
    }

    private static void exceptionMapper(Exception e) throws DeeplinkException {
        if (e instanceof ConnectException || e instanceof SocketTimeoutException || e instanceof UnknownHostException) {
            throwCustomerException(AMKDLErrorCode.NO_INTERNET_CONNECTION);
        } else {
            //default error
            throwCustomerException(AMKDLErrorCode.SOMETHING_WENT_WRONG);
        }
    }


    private static <T> AMKDLResponse<AMKDeeplinkData> okHttpClientGenerator(T request, boolean isIndividual) throws DeeplinkException {
        String endpoint;
        if (isIndividual) {
            endpoint = "/deep-link/generate/individual";
        } else endpoint = "/deep-link/generate/merchant";
        String accessToken = getAccessToken();

        String requestBody;
        String referenceId;
        if (request instanceof AMKDLIndividual) {
            IndividualRequest individualRequest = AMKRequestBody.getIndividualRequest((AMKDLIndividual) request);
            referenceId = individualRequest.getReferenceId();
            requestBody = gson.toJson(individualRequest);
        } else {
            MerchantRequest merchantRequest = AMKRequestBody.getMerchantRequest((AMKDLMerchant) request);
            referenceId = merchantRequest.getReferenceId();
            requestBody = gson.toJson(merchantRequest);
        }

        RequestBody body = RequestBody.create(mediaType, requestBody);
        Request apiRequest = new Request.Builder()
                .url(BASE_URL + endpoint)
                .method("POST", body)
                .addHeader("Authorization", "Bearer " + accessToken)
                .build();
        try {
            Response response = client.newCall(apiRequest).execute();
            return convertToDeeplinkResponseBody(response, referenceId);
        } catch (Exception e) {
            exceptionMapper(e);
        }
        return null;
    }

    @NotNull
    private static String getAccessToken() throws DeeplinkException {
        String accessToken = null;
        try {
            accessToken = AuthenticationPresenter.getAccessToken(mClientId, mSecretKey, mJwtAudience);
        } catch (Exception e) {
            exceptionHandler(e);
        }
        if (accessToken == null) throwCustomerException(AMKDLErrorCode.JWT_CONFIG_IS_INVALID);
        return accessToken;
    }

    private static void exceptionHandler(Exception e) throws DeeplinkException {
        if (e instanceof DeeplinkException) {
            throwCustomerException(((DeeplinkException) e).getErrorCode());
        } else {
            throwCustomerException(AMKDLErrorCode.SOMETHING_WENT_WRONG);
        }
    }
}
