package amk.sdk.deeplink.entity.model;

import amk.sdk.deeplink.exception.DeeplinkException;
import amk.sdk.deeplink.utils.StringUtils;

import java.math.BigDecimal;

import static amk.sdk.deeplink.exception.DeeplinkException.throwCustomerException;
import static amk.sdk.deeplink.entity.model.AMKDLErrorCode.*;
import static amk.sdk.deeplink.utils.StringUtils.isBlank;

public class AMKDLValidation {

    private static boolean isAmountInvalid(Double amount, String currency) {
        if (amount != null) {

            BigDecimal bd = BigDecimal.valueOf(amount);
            String valueAmount = bd.stripTrailingZeros().toPlainString();
            int index = valueAmount.indexOf(".");
            int decimalCount = index < 0 ? 0 : valueAmount.length() - index - 1;

            int AMOUNT_LENGTH = 13;
            if (amount < 0 || valueAmount.length() > AMOUNT_LENGTH) {
                return true;
            }
            if (currency.equalsIgnoreCase("KHR")) {
                //0
                return decimalCount != 0;
            } else {
                //0.##
                return decimalCount >= 3;
            }
        } else return false;
    }

    public static String getReceiverAccountId(String accountId) throws DeeplinkException {
        int ACCOUNT_ID_LENGTH = 32;
        if (isBlank(accountId)) {
            throwCustomerException(ACCOUNT_ID_REQUIRED);
        } else if (accountId.length() > ACCOUNT_ID_LENGTH) {
            throwCustomerException(ACCOUNT_ID_LENGTH_INVALID);
        }
        return accountId;
    }

    public static String getMerchantName(String merchantName) throws DeeplinkException {
        int MERCHANT_NAME_LENGTH = 25;
        if (isBlank(merchantName)) {
            throwCustomerException(MERCHANT_NAME_REQUIRED);
        } else if (merchantName.length() > MERCHANT_NAME_LENGTH) {
            throwCustomerException(MERCHANT_NAME_LENGTH_INVALID);
        }
        return merchantName;
    }

    public static String getReceiverName(String receiverName) throws DeeplinkException {
        return receiverName;
    }

    public static String getUsername(String username) throws DeeplinkException {
        int MERCHANT_NAME_LENGTH = 25;
        if (isBlank(username)) {
            throwCustomerException(USERNAME_REQUIRED);
        } else if (username.length() > MERCHANT_NAME_LENGTH) {
            throwCustomerException(USERNAME_LENGTH_INVALID);
        }
        return username;
    }

    public static String getMerchantAccount(String merchantId) throws DeeplinkException {
        int MERCHANT_ID_LENGTH = 25;
        if (isBlank(merchantId)) {
            throwCustomerException(MERCHANT_ID_REQUIRED);
        } else if (merchantId.length() > MERCHANT_ID_LENGTH) {
            throwCustomerException(MERCHANT_ID_LENGTH_INVALID);
        }
        return merchantId;
    }

    public static String getCurrencyType(String currency) throws DeeplinkException {
        if (StringUtils.isBlank(currency)) {
            throwCustomerException(CURRENCY_TYPE_REQUIRED);
        }
        return currency;
    }

    public static Double getAmount(Double amount, String currency) throws DeeplinkException {
        if (isAmountInvalid(amount, currency)) {
            throwCustomerException(AMOUNT_INVALID);
        }
        return amount;
    }

    public static String getMobileNumber(String mobileNumber) throws DeeplinkException {
        int MOBILE_NUMBER_LENGTH = 12;
        if (mobileNumber != null && mobileNumber.length() > MOBILE_NUMBER_LENGTH) {
            throwCustomerException(MOBILE_NUMBER_LENGTH_INVALID);
        }
        return mobileNumber;
    }

    public static String getBillNumber(String billNumber) throws DeeplinkException {
        int BILL_NUMBER_LENGTH = 25;
        if (billNumber != null && billNumber.length() > BILL_NUMBER_LENGTH) {
            throwCustomerException(AMKDLErrorCode.BILL_NUMBER_LENGTH_INVALID);
        }
        return billNumber;
    }

    public static String getStoreLabel(String storeLabel) throws DeeplinkException {
        int STORE_LABEL_LENGTH = 25;
        if (storeLabel != null && storeLabel.length() > STORE_LABEL_LENGTH) {
            throwCustomerException(STORE_LABEL_LENGTH_INVALID);
        }
        return storeLabel;
    }

    public static String getTerminalLabel(String terminalLabel) throws DeeplinkException {
        int TERMINAL_LABEL_LENGTH = 25;
        if (terminalLabel != null && terminalLabel.length() > TERMINAL_LABEL_LENGTH) {
            throwCustomerException(TERMINAL_LABEL_LENGTH_INVALID);
        }
        return terminalLabel;
    }

    public static SourceInfo getSourceInfo(SourceInfo sourceInfo) throws DeeplinkException {
        if (sourceInfo != null) {
            String appName = sourceInfo.getAppName();
            String appDeepLinkCallback = sourceInfo.getAppDeeplinkCallback();
            String appIconUrl = sourceInfo.getAppIconUrl();
            if (isBlank(appName) || isBlank(appDeepLinkCallback) || isBlank(appIconUrl))
                throwCustomerException(SOURCE_INFO_INVALID);
        }
        return sourceInfo;
    }


    public static String getReferenceId(String referenceId, String id) {
        if (isBlank(referenceId)) {
            return Md5Config.getDefaultHash(id);
        } else return referenceId;
    }

    public static String getValidReferenceId(String referenceId) throws DeeplinkException {
        if (isBlank(referenceId)) {
            throwCustomerException(REFERENCE_ID_REQUIRED);
        }
        return referenceId;
    }

    public static String getMerchantCity(String merchantCity) throws DeeplinkException {
        int MERCHANT_CITY_LENGTH = 15;
        if (merchantCity != null && merchantCity.length() > MERCHANT_CITY_LENGTH) {
            throwCustomerException(CITY_LENGTH_INVALID);
        }
        return merchantCity;
    }
}
