package amk.sdk.deeplink.presenter;

import amk.sdk.deeplink.utils.StringUtils;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

import java.nio.charset.StandardCharsets;
import java.security.KeyFactory;
import java.security.PrivateKey;
import java.security.spec.EncodedKeySpec;
import java.security.spec.PKCS8EncodedKeySpec;
import java.util.Base64;
import java.util.Date;

import static amk.sdk.deeplink.entity.model.AMKDLErrorCode.*;
import static amk.sdk.deeplink.exception.DeeplinkException.throwCustomerException;

final class JwtTokenConfigPresenter {

    private static final long JWT_TOKEN_VALIDITY = 300000;
    private static final long JWT_TOKEN_OFFSET = (10 * 60 * 1000);

    public static String generateJwtToken(String clientId, String secretKey, String jwtAudience) throws Exception {

        if (StringUtils.isBlank(clientId)) {
            throwCustomerException(CLIENT_ID_REQUIRED);
        } else if (StringUtils.isBlank(secretKey)) {
            throwCustomerException(SECRET_KEY_REQUIRED);
        } else if (StringUtils.isBlank(jwtAudience)) {
            throwCustomerException(JWT_AUDIENCE_REQUIRED);
        }

        long nowMillis = System.currentTimeMillis() - JWT_TOKEN_OFFSET;
        Date now = new Date(nowMillis);
        long expired = nowMillis + JWT_TOKEN_VALIDITY + JWT_TOKEN_OFFSET;
        Date exp = new Date(expired);
        return Jwts.builder()
                .setIssuer(clientId)
                .setAudience(jwtAudience)
                .setIssuedAt(now)
                .setExpiration(exp)
                // RS256 with privateKey
                .signWith(SignatureAlgorithm.RS256, getRawSecretKey(secretKey))
                .compact();

    }

    private static PrivateKey getRawSecretKey(String secretKey) throws Exception {
        String replaceSecretKeyString = secretKey
                .replaceAll(System.lineSeparator(), "")
                .replaceAll("\r", "")
                .replace("-----BEGIN RSA PRIVATE KEY-----", "")
                .replace("-----END RSA PRIVATE KEY-----", "")
                .replace("-----BEGIN PRIVATE KEY-----", "")
                .replace("-----END PRIVATE KEY-----", "");
        KeyFactory fact = KeyFactory.getInstance("RSA");
        byte[] bytes = Base64.getDecoder().decode(replaceSecretKeyString.getBytes(StandardCharsets.UTF_8));
        EncodedKeySpec keySpec = new PKCS8EncodedKeySpec(bytes);
        return fact.generatePrivate(keySpec);
    }
}
