package amk.sdk.deeplink.presenter;

import amk.sdk.deeplink.entity.model.*;
import amk.sdk.deeplink.entity.response.AMKDLResponse;
import amk.sdk.deeplink.exception.DeeplinkException;
import okhttp3.*;
import org.jetbrains.annotations.NotNull;
import org.json.JSONException;
import org.json.JSONObject;

import java.net.ConnectException;
import java.net.SocketTimeoutException;
import java.net.UnknownHostException;

public final class DeeplinkPresenter {
    private static String mClientId;
    private static String mSecretKey;
    private static String mJwtAudience = "http://apiserver/api/oauth/token";
    private static final OkHttpClient client = new OkHttpClient().newBuilder().build();
    private static final MediaType mediaType = MediaType.parse("application/json");

    public static void preConfig(String clientId, String secretKey, String jwtAudience) {
        mClientId = clientId;
        mSecretKey = secretKey;
        mJwtAudience = jwtAudience;
        ApiConstant.reset();
    }

    public static void preConfig(String baseUrl) {
        if(baseUrl.equals(ApiConstant.getBaseUrl())) {
        	ApiConstant.reset();
        }else {
        	ApiConstant.switchUrl();
        }
    }
    public static void preConfig(String clientId, String secretKey) {
        preConfig(clientId,secretKey,"http://apiserver/api/oauth/token");
    }
    
    public static String generateJwt(String clientId, String secretKey, String jwtAudience) {
        try {
            return JwtTokenConfigPresenter.generateJwtToken(clientId, secretKey, jwtAudience);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static AMKDLResponse<AMKDeeplinkData> generateIndividualDeeplink(AMKDLIndividual amkdlIndividual) throws DeeplinkException {
        return okHttpClientGenerator(amkdlIndividual);

    }

    public static AMKDLResponse<AMKDeeplinkData> generateIndividualDeeplinkV2(AMKDLIndividual amkdlIndividual,String accessToken) throws DeeplinkException {
        return okHttpClientGeneratorV2(amkdlIndividual,accessToken);
    }

    public static AMKDLResponse<AMKDeeplinkData> generateMerchantDeeplink(AMKDLMerchant amkdlMerchant) throws DeeplinkException {
        return okHttpClientGenerator(amkdlMerchant);

    }

    public static AMKDLResponse<AMKPaymentStatusData> checkPaymentStatus(String referenceId) throws DeeplinkException {
        String endpoint = "deep-link/payment/status";
        String accessToken = getAccessToken();
        JSONObject bodyRequest = new JSONObject();
        try {
            bodyRequest.put("referenceId", referenceId);
        } catch (JSONException ignored) {
        }
        String requestBody = bodyRequest.toString();
        RequestBody body = RequestBody.create(requestBody, mediaType);
        Request request = new Request.Builder()
                .url(ApiConstant.getBaseUrl() + endpoint)
                .method("POST", body)
                .addHeader("Authorization", "Bearer " + accessToken)
                .build();
        try {
            Response response = client.newCall(request).execute();
            return PaymentStatusResponseMapper.convertToPaymentStatusResponseBody(response);
        } catch (Exception e) {
            exceptionMapper(e);
        }
        return null;
    }

    private static void exceptionMapper(Exception e) throws DeeplinkException {
        if (e instanceof ConnectException || e instanceof SocketTimeoutException || e instanceof UnknownHostException) {
            DeeplinkException.throwCustomerException(AMKDLErrorCode.NO_INTERNET_CONNECTION);
        } else {
            //default error
            DeeplinkException.throwCustomerException(AMKDLErrorCode.SOMETHING_WENT_WRONG);
        }
    }

    private static <T> AMKDLResponse<AMKDeeplinkData> okHttpClientGenerator(AMKRequestBody request) throws DeeplinkException {
        String endpoint = request.getRequestEndpoint();
        String accessToken = getAccessToken();
        String requestBody = request.toString();
        String referenceId = request.getReferenceId();
        RequestBody body = RequestBody.create(requestBody, mediaType);
        Request apiRequest = new Request.Builder()
                .url(ApiConstant.getBaseUrl() + endpoint)
                .method("POST", body)
                .addHeader("Authorization", "Bearer " + accessToken)
                .build();
        try {
            Response response = client.newCall(apiRequest).execute();
            return DeeplinkResponseMapper.convertToDeeplinkResponseBody(response, referenceId);
        } catch (Exception e) {
            exceptionMapper(e);
        }
        return null;
    }
    private static <T> AMKDLResponse<AMKDeeplinkData> okHttpClientGeneratorV2(AMKRequestBody request,String accessToken) throws DeeplinkException {
        String endpoint = request.getRequestEndpoint();
        String requestBody = request.toString();
        String referenceId = request.getReferenceId();
        RequestBody body = RequestBody.create(requestBody, mediaType);
        Request apiRequest = new Request.Builder()
                .url(ApiConstant.getBaseUrl() + endpoint)
                .method("POST", body)
                .addHeader("Authorization", "Bearer " + accessToken)
                .build();
        try {
            Response response = client.newCall(apiRequest).execute();
            return DeeplinkResponseMapper.convertToDeeplinkResponseBody(response, referenceId);
        } catch (Exception e) {
            exceptionMapper(e);
        }
        return null;
    }

    @NotNull
    private static String getAccessToken() throws DeeplinkException {
        String accessToken = null;
        try {
            accessToken = AuthenticationPresenter.getAccessToken(mClientId, mSecretKey, mJwtAudience);
        } catch (Exception e) {
            exceptionHandler(e);
        }
        if (accessToken == null) DeeplinkException.throwCustomerException(AMKDLErrorCode.JWT_CONFIG_IS_INVALID);
        return accessToken;
    }

    private static void exceptionHandler(Exception e) throws DeeplinkException {
        if (e instanceof DeeplinkException) {
            DeeplinkException.throwCustomerException(((DeeplinkException) e).getErrorCode());
        } else {
            DeeplinkException.throwCustomerException(AMKDLErrorCode.SOMETHING_WENT_WRONG);
        }
    }
}
