package com.sentray.kmmprotocolmodule.network.tcp

import com.sentray.kmmprotocolmodule.sdk.parser.Parser
import com.sentray.kmmprotocolmodule.utilityToolKit.genericClasses.KmmTimer
import com.sentray.kmmprotocolmodule.utilityToolKit.loggerKit.KMMLogger
import io.ktor.network.selector.*
import io.ktor.network.sockets.*
import io.ktor.utils.io.*
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlin.time.Duration.Companion.seconds

//例子：https://gitee.com/slientes/custom-ktor
internal class TcpClient(private val ipAddress: String, private val port: Int) {
    internal enum class ETcpSocketStatus {
        Connected,
        Disconnected,
        Connecting,
    }

    private var selectorManager: SelectorManager? = null
    private var tcpSocket: Socket? = null
    private var sendChannel: ByteWriteChannel? = null
    private var reConnectTimer: KmmTimer? = null
    private val reConnectCheckDuration = 3.seconds
    private var status = ETcpSocketStatus.Disconnected

    private val parser = Parser()

    suspend fun startConnect(): ETcpSocketStatus {
        //若已连接，则先关闭
        disconnect()

        if (tcpSocket == null) {
            status = ETcpSocketStatus.Connecting
            try {
                selectorManager = SelectorManager(Dispatchers.Default)
                selectorManager?.let {
                    tcpSocket = aSocket(it).tcp().connect(ipAddress, port)
                }
                //若连接成功，则处理输入和输出事宜
                tcpSocket?.let { tcpSocketValid ->
                    val receiveChannel = tcpSocketValid.openReadChannel()
                    sendChannel = tcpSocketValid.openWriteChannel(autoFlush = true)

                    KMMLogger.d("TCP Server connect success")

                    status = ETcpSocketStatus.Connected
                    startReconnectTimer()

                    CoroutineScope(Dispatchers.Default).launch {
                        while (tcpSocket != null) {
                            if (!receiveChannel.isClosedForRead) {
                                KMMLogger.d("TCP Client Ready : start receive data")
                                receiveChannel.awaitContent()
                                if (receiveChannel.availableForRead == 0) break
                                val byteArray = ByteArray(receiveChannel.availableForRead)
                                receiveChannel.readAvailable(byteArray)
                                val receiveMessage = byteArray.decodeToString()
                                KMMLogger.d("TCP Client Receive : $receiveMessage")
                                val parserResultList = parser.parse(receiveMessage)
                                //TODO: 结果缓存至 database
                            } else {
                                disconnect()
                                KMMLogger.d("TCP Client Failure : Read Channel failure")
                            }
                        }
                    }
                }
            } catch (e: Exception) {
                status = ETcpSocketStatus.Disconnected
            }
        }

        return status
    }

    suspend fun sendData(message: String) {
        if (status == ETcpSocketStatus.Connected) {
            sendChannel?.let {
                if (!it.isClosedForWrite) {
                    try {
                        it.writeStringUtf8(message)
                        KMMLogger.d("TCP Client send data : $message")
                    } catch (e: Exception) {
                        disconnect()
                        e.printStackTrace()
                    }
                } else {
                    KMMLogger.d("TCP Client send data failure, closed already: $message")
                    disconnect()
                }
            } ?: run {
                KMMLogger.d("TCP Client Not Connect")
                status = ETcpSocketStatus.Disconnected
            }
        } else {
            KMMLogger.d("TCP Client Status : $status")
        }
    }

    private fun disconnect() {
        tcpSocket?.close()
        tcpSocket = null

        selectorManager?.close()
        selectorManager = null

        sendChannel?.close()
        sendChannel = null

        status = ETcpSocketStatus.Disconnected
    }

    fun disconnectManually() {
        stopReconnectTimer()
        disconnect()
        KMMLogger.d("TCP Client Disconnect success")
    }

    private fun startReconnectTimer() {
        stopReconnectTimer()
        if (reConnectTimer == null) {
            reConnectTimer = KmmTimer().also {
                it.startScheduleTimer(reConnectCheckDuration) {
                    if (status == ETcpSocketStatus.Disconnected) {
                        CoroutineScope(Dispatchers.Default).launch {
                            KMMLogger.d("TCP 链接断开，尝试重连")
                            startConnect()
                            KMMLogger.d("TCP 重连结果：$status")
                        }
                    }
                    KMMLogger.d("checking tcp connection : ${tcpSocket?.isClosed} : ${sendChannel?.isClosedForWrite} : ${selectorManager?.isActive}")
                }
            }
        }
    }

    private fun stopReconnectTimer() {
        reConnectTimer?.stopScheduleTimer()
        reConnectTimer = null
    }
}