package com.sentray.kmmprotocolmodule.sdk.command

import com.sentray.kmmprotocolmodule.sdk.parser.parserResult.ControllableApplianceInfo
import com.sentray.kmmprotocolmodule.sdk.parser.parserResult.DetectableApplianceInfo
import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeClassify
import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeSet
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ENetworkType
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ProtocolVersion
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.ParaStructure
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolCreator.CommandData
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolCreator.ControlCmdCreator
import kotlin.jvm.JvmStatic

//控制命令
class ControlCmd {
    enum class ControlAction {
        On,
        Off
    }

    companion object {
        @JvmStatic
        fun controlApplianceCmd(
            sn: String,
            deviceType: String,
            jackIndex: Int,
            controlAction: ControlAction,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData? {
            val commandData = DeviceTypeSet.fromRaw(deviceType)?.let { deviceTypeSet ->
                val controllableApplianceInfo =
                    ControllableApplianceInfo(sn, deviceTypeSet, jackIndex)
                controlApplianceCmd(
                    controllableApplianceInfo,
                    controlAction,
                    protocolVersion,
                    networkType
                )
            }

            return commandData
        }

        //控制命令
        @JvmStatic
        fun controlApplianceCmd(
            controllableApplianceInfo: ControllableApplianceInfo,
            controlAction: ControlAction,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData? {
            val action = when (controlAction) {
                ControlAction.On -> true
                ControlAction.Off -> false
            }

            val commandData = when {
                DeviceTypeClassify.isControllableDevice(controllableApplianceInfo.devType) -> ControlCmdCreator.controlApplianceOnOff(
                    controllableApplianceInfo.sn,
                    controllableApplianceInfo.devType,
                    controllableApplianceInfo.applianceIndex,
                    action,
                    protocolVersion = protocolVersion,
                    networkType = networkType
                )
                else -> null
            }

            return commandData
        }

        @JvmStatic
        fun controlApplianceGroupCmd(
            controllableApplianceInfoList: List<ControllableApplianceInfo>,
            controlAction: ControlAction,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): List<CommandData> {
            val cmdDataList = mutableListOf<CommandData>()

            val action = when (controlAction) {
                ControlAction.On -> true
                ControlAction.Off -> false
            }

            //1. 按照 sn 进行分类
            val snSet = mutableSetOf<String>()
            controllableApplianceInfoList.forEach {
                snSet.add(it.sn)
            }
            val controllableApplianceInfoListTemp =
                controllableApplianceInfoList.toMutableList()

            snSet.forEach { sn ->
                val controllableApplianceInfoListForOneSN =
                    mutableListOf<ControllableApplianceInfo>()
                controllableApplianceInfoListTemp.forEach {
                    if (it.sn == sn) {
                        controllableApplianceInfoListForOneSN.add(it)
                    }
                }
                controllableApplianceInfoListTemp.removeAll(
                    controllableApplianceInfoListForOneSN
                )

                //2. 按照 deviceType 进行分类
                val deviceTypeGroupSet = mutableSetOf<DeviceTypeSet>()
                controllableApplianceInfoListForOneSN.forEach {
                    deviceTypeGroupSet.add(it.devType)
                }

                deviceTypeGroupSet.forEach { deviceTypeSet ->
                    val controllableApplianceInfoListForOneDeviceType =
                        mutableListOf<ControllableApplianceInfo>()

                    controllableApplianceInfoListForOneSN.forEach {
                        if (deviceTypeSet == it.devType &&
                            DeviceTypeClassify.isControllableDevice(it.devType)
                        ) {
                            controllableApplianceInfoListForOneDeviceType.add(it)
                        }
                    }
                    if (controllableApplianceInfoListForOneDeviceType.count() > 0) {
                        controllableApplianceInfoListForOneSN.removeAll(
                            controllableApplianceInfoListForOneDeviceType
                        )

                        val firstObject = controllableApplianceInfoListForOneDeviceType[0]

                        val applianceIndexSet = mutableSetOf<Int>()
                        controllableApplianceInfoListForOneDeviceType.forEach {
                            applianceIndexSet.add(it.applianceIndex)
                        }

                        //3. 生成命令
                        cmdDataList.add(
                            ControlCmdCreator.controlApplianceGroupOnOff(
                                firstObject.sn,
                                firstObject.devType,
                                applianceIndexSet.toList(),
                                action,
                                protocolVersion = protocolVersion,
                                networkType = networkType
                            )
                        )
                    }
                }
            }

            return cmdDataList
        }

        //设置命令
        @JvmStatic
        fun adjustableLightControlCmd(
            controllableApplianceInfo: ControllableApplianceInfo,
            para: ParaStructure,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData? {
            val cmdData = when {
                DeviceTypeClassify.isDimmerDevice(controllableApplianceInfo.devType) -> {
                    ControlCmdCreator.brightAdjust(
                        controllableApplianceInfo.sn,
                        controllableApplianceInfo.devType,
                        controllableApplianceInfo.applianceIndex,
                        para.getRawValue().toInt(),
                        protocolVersion = protocolVersion,
                        networkType = networkType
                    )
                }
                DeviceTypeClassify.isTemperatureLampDevice(controllableApplianceInfo.devType) -> {
                    ControlCmdCreator.temperatureParaAdjust(
                        controllableApplianceInfo.sn,
                        controllableApplianceInfo.devType,
                        controllableApplianceInfo.applianceIndex,
                        para.getTemperatureLampBright().toInt(),
                        para.getTemperatureLampTemp().toInt(),
                        protocolVersion = protocolVersion,
                        networkType = networkType
                    )
                }
                DeviceTypeClassify.isRGBLampDevice(controllableApplianceInfo.devType) || DeviceTypeClassify.isRGBWLampDevice(
                    controllableApplianceInfo.devType
                ) -> {
                    ControlCmdCreator.rgbwParaAdjust(
                        controllableApplianceInfo.sn,
                        controllableApplianceInfo.devType,
                        controllableApplianceInfo.applianceIndex,
                        para,
                        protocolVersion = protocolVersion,
                        networkType = networkType
                    )
                }
                else -> {
                    null
                }
            }

            return cmdData
        }

        @JvmStatic
        fun irControlCmd(
            controllableApplianceInfo: ControllableApplianceInfo,
            irCmd: ParaStructure.IrCmd,
            irIndex: Int,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData? {
            return when {
                DeviceTypeClassify.isIRDevice(controllableApplianceInfo.devType) ->
                    ControlCmdCreator.irControl(
                        controllableApplianceInfo.sn,
                        controllableApplianceInfo.devType,
                        controllableApplianceInfo.applianceIndex,
                        irCmd,
                        irIndex,
                        protocolVersion = protocolVersion,
                        networkType = networkType
                    )
                else -> null
            }
        }

        @JvmStatic
        fun curtainControlCmd(
            controllableApplianceInfo: ControllableApplianceInfo,
            curtainAction: ParaStructure.CurtainAction,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData? {
            return when {
                DeviceTypeClassify.isCurtainWithoutLocationDevice(controllableApplianceInfo.devType) -> ControlCmdCreator.curtainControl(
                    controllableApplianceInfo.sn,
                    controllableApplianceInfo.devType,
                    controllableApplianceInfo.applianceIndex,
                    curtainAction,
                    protocolVersion = protocolVersion,
                    networkType = networkType
                )
                else -> null
            }
        }

        @JvmStatic
        fun curtainWithLocationPercentageControlCmd(
            controllableApplianceInfo: ControllableApplianceInfo,
            curtainLocationPercentage: Int,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData? {
            return when {
                DeviceTypeClassify.isCurtainWithLocationDevice(controllableApplianceInfo.devType) -> ControlCmdCreator.curtainWithLocationPercentageControl(
                    controllableApplianceInfo.sn,
                    controllableApplianceInfo.devType,
                    controllableApplianceInfo.applianceIndex,
                    curtainLocationPercentage,
                    protocolVersion = protocolVersion,
                    networkType = networkType
                )
                else -> null
            }
        }

        @JvmStatic
        fun curtainWithLocationActionControlCmd(
            controllableApplianceInfo: ControllableApplianceInfo,
            curtainWithLocAction: ParaStructure.CurtainWithLocAction,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData? {
            return when {
                DeviceTypeClassify.isCurtainWithLocationDevice(controllableApplianceInfo.devType) -> ControlCmdCreator.curtainWithLocationActionControl(
                    controllableApplianceInfo.sn,
                    controllableApplianceInfo.devType,
                    controllableApplianceInfo.applianceIndex,
                    curtainWithLocAction,
                    protocolVersion = protocolVersion,
                    networkType = networkType
                )
                else -> null
            }
        }

        @JvmStatic
        fun activateDetectableApplianceCmd(
            detectableApplianceInfo: DetectableApplianceInfo,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData? {
            val devType = detectableApplianceInfo.devType ?: return null
            val applianceIndex = detectableApplianceInfo.applianceIndex ?: return null

            return when {
                DeviceTypeClassify.isInputDetectDevice(devType) -> {
                    ControlCmdCreator.activateTrigger(
                        detectableApplianceInfo.sn,
                        applianceIndex,
                        protocolVersion = protocolVersion,
                        networkType = networkType
                    )
                }
                else -> {
                    null
                }
            }
        }
    }
}