package com.sentray.kmmprotocolmodule.tcpProtocol.protocolCreator

import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeSet
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.*
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.TimerParaV1
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.TimerParaV2
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.WeekMask
import com.sentray.kmmprotocolmodule.utilityToolKit.HexToolKit
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlin.jvm.JvmStatic

internal class TimerCmdCreator {
    companion object {
        //Timer V1
        @Serializable
        private data class GetTimerV1Para(
            val sn: String,
            val seq: Int,
            val factoryID: Int,
            val cmd: String,
            val field: List<String>,
        )

        @JvmStatic
        //获取定时器 V1 版本
        fun getTimerV1(
            sn: String,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val para = GetTimerV1Para(
                sn,
                seq,
                factoryID.raw,
                Cmd.Get.raw,
                listOf(ProtocolKey.Timer.raw),
            )

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(para)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @Serializable
        data class AddOrModifyTimerV1Para(
            val sn: String,
            val seq: Int,
            val factoryID: Int,
            val cmd: String,
            val timer: List<String>,
        )

        @Deprecated("TimerV1 已弃用")
        @JvmStatic
        fun addTimerV1(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            action: Boolean,
            enable: Boolean,
            timestamp: Long,
            weekMaskFromUserTimezone: WeekMask,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val timerParaV1 = TimerParaV1()
            timerParaV1.timerBasePara.timerIndex = 0
            timerParaV1.timerBasePara.posix = timestamp
            timerParaV1.timerBasePara.devType = deviceTypeSet
            timerParaV1.timerBasePara.enable = enable
            timerParaV1.timerBasePara.setWeekMaskForUTC0(weekMaskFromUserTimezone)
            HexToolKit.hexStringToBoolList(HexToolKit.indexToHexString(applianceIndex))
                ?.let { rlyMaskBoolList ->
                    timerParaV1.rlyMask = rlyMaskBoolList
                }

            timerParaV1.actionMask = when (action) {
                true -> timerParaV1.rlyMask
                false -> listOf(false)
            }

            val para = AddOrModifyTimerV1Para(
                sn,
                seq,
                factoryID.raw,
                Cmd.AddTimer.raw,
                listOf(timerParaV1.toTimerParaV1String())
            )

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(para)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @Deprecated("TimerV1 已弃用")
        @JvmStatic
        fun modifyTimerV1(
            sn: String,
            timerIndex: Int,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            action: Boolean,
            enable: Boolean,
            timestamp: Long,    //TODO: 一次性定时器，enable 时，timestamp若比当前时间早，则需重新计算
            weekMaskFromUserTimezone: WeekMask,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val timerParaV1 = TimerParaV1()
            timerParaV1.timerBasePara.timerIndex = timerIndex
            timerParaV1.timerBasePara.posix = timestamp
            timerParaV1.timerBasePara.devType = deviceTypeSet
            timerParaV1.timerBasePara.enable = enable
            timerParaV1.timerBasePara.setWeekMaskForUTC0(weekMaskFromUserTimezone)
            HexToolKit.hexStringToBoolList(HexToolKit.indexToHexString(applianceIndex))
                ?.let { rlyMaskBoolList ->
                    timerParaV1.rlyMask = rlyMaskBoolList
                }

            timerParaV1.actionMask = when (action) {
                true -> timerParaV1.rlyMask
                false -> listOf(false)
            }

            val para = AddOrModifyTimerV1Para(
                sn,
                seq,
                factoryID.raw,
                Cmd.ModifyTimer.raw,
                listOf(timerParaV1.toTimerParaV1String())
            )

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(para)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }


        @Serializable
        data class DelTimerV1Para(
            val sn: String,
            val seq: Int,
            val factoryID: Int,
            val cmd: String,
            val timer: List<Int>,
        )

        @Deprecated("TimerV1 已弃用")
        @JvmStatic
        fun deleteTimerV1(
            sn: String,
            timerIndex: Int,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val para = DelTimerV1Para(
                sn,
                seq,
                factoryID.raw,
                Cmd.DelTimer.raw,
                listOf(timerIndex)
            )

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(para)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }


        //Timer V2
        @Serializable
        data class AddOrModifyTimerV2Para(
            val sn: String,
            val seq: Int,
            val factoryID: Int,
            val cmd: String,
            val timerPara: List<Long>,
        )

        @JvmStatic
        fun addTimerV2(
            sn: String,
            actionMaskList: List<Boolean>,
            triggerStatusList: List<Boolean>,
            triggerMaskList: List<Boolean>,
            enable: Boolean,
            timestamp: Long,
            weekMaskFromUserTimezone: WeekMask,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val timerParaV2 = TimerParaV2()
            timerParaV2.timerBasePara.timerIndex = 0
            timerParaV2.timerBasePara.enable = enable
            timerParaV2.timerBasePara.posix = timestamp
            timerParaV2.timerBasePara.setWeekMaskForUTC0(weekMaskFromUserTimezone)
            timerParaV2.userActionMask = actionMaskList
            timerParaV2.triggerStatus = triggerStatusList
            timerParaV2.triggerMask = triggerMaskList


            val seq = Util.randomSeq()
            val para = AddOrModifyTimerV2Para(
                sn,
                seq,
                factoryID.raw,
                Cmd.AddTimer2.raw,
                timerParaV2.toParaList()
            )

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(para)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun modifyTimerV2(
            sn: String,
            timerIndex: Int,
            actionMaskList: List<Boolean>,
            triggerStatusList: List<Boolean>,
            triggerMaskList: List<Boolean>,
            enable: Boolean,
            timestamp: Long,
            weekMaskFromUserTimezone: WeekMask,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val timerParaV2 = TimerParaV2()
            timerParaV2.timerBasePara.timerIndex = timerIndex
            timerParaV2.timerBasePara.enable = enable
            timerParaV2.timerBasePara.posix = timestamp
            timerParaV2.timerBasePara.setWeekMaskForUTC0(weekMaskFromUserTimezone)
            timerParaV2.userActionMask = actionMaskList
            timerParaV2.triggerStatus = triggerStatusList
            timerParaV2.triggerMask = triggerMaskList

            val seq = Util.randomSeq()
            val para = AddOrModifyTimerV2Para(
                sn,
                seq,
                factoryID.raw,
                Cmd.ModifyTimer2.raw,
                timerParaV2.toParaList()
            )

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(para)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @Serializable
        data class DelTimerV2Para(
            val sn: String,
            val seq: Int,
            val factoryID: Int,
            val cmd: String,
            val timerPara: List<Int>,
        )

        @JvmStatic
        fun delTimerV2(
            sn: String,
            timerIndex: Int,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val para = DelTimerV2Para(
                sn,
                seq,
                factoryID.raw,
                Cmd.DelTimer2.raw,
                listOf(timerIndex)
            )

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(para)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }
    }
}