@file:OptIn(ExperimentalForeignApi::class, ExperimentalForeignApi::class)

package com.sentray.kmmprotocolmodule.network.udp

import cocoapods.CocoaAsyncSocket.GCDAsyncUdpSocket
import cocoapods.CocoaAsyncSocket.GCDAsyncUdpSocketDelegateProtocol
import kotlinx.cinterop.ExperimentalForeignApi
import platform.Foundation.*
import platform.darwin.NSObject
import platform.darwin.dispatch_get_global_queue

internal actual class PlatformUdpClient actual constructor() {
    private class InternalUdpClient(
        private val host: String,
        private val port: Int,
        private val receiveCallback: (message: String) -> Unit
    ) {
        var udpSocket: GCDAsyncUdpSocket? = null
        var sendTag = 0

        fun connect() {
            val delegate =
                object : NSObject(), GCDAsyncUdpSocketDelegateProtocol {
                    override fun udpSocket(
                        sock: GCDAsyncUdpSocket,
                        didReceiveData: NSData,
                        fromAddress: NSData,
                        withFilterContext: Any?
                    ) {
                        NSString.create(didReceiveData, NSUTF8StringEncoding)?.let { nsString ->
                            val string = nsString as String
                            receiveCallback(string)
                        }
                    }
                }


            if (udpSocket == null) {
                val delegateQueue = platform.darwin.dispatch_queue_create(
                    "UDP Socket Delegate Queue",
                    dispatch_get_global_queue(1, 0u)
                )
                udpSocket = GCDAsyncUdpSocket(
                    delegate = delegate,
                    delegateQueue = delegateQueue
                )
            }

            udpSocket?.let {
                try {
                    it.setIPv4Enabled(true)
                    it.setIPv6Enabled(true)
                    it.enableReusePort(true, null)
                    it.enableBroadcast(true, null)
                    it.beginReceiving(null)
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        }

        fun send(message: String) {
            udpSocket?.let { socket ->
                val nsMessage = NSString.create(string = message)
                val nsPort = port.toUShort()
                val nsTimeInterval = (-1).toDouble()
                nsMessage.dataUsingEncoding(NSUTF8StringEncoding)?.let { nsData ->
                    socket.sendData(
                        data = nsData,
                        toHost = host,
                        port = nsPort,
                        withTimeout = nsTimeInterval,
                        tag = sendTag.toLong()
                    )
                    sendTag++
                }
            }
        }

        fun disconnect() {
            udpSocket?.close()
        }
    }


    private var udpClient: InternalUdpClient? = null
    private var receiveCallback: ((message: String) -> Unit)? = null

    actual fun broadcastPackage(
        broadcastAddress: String,
        broadcastPort: Int,
        broadcastMessage: String
    ): Boolean {
        if (udpClient == null) {
            udpClient = InternalUdpClient(
                host = broadcastAddress,
                port = broadcastPort
            ) { messageReceive: String ->
                receiveCallback?.invoke(messageReceive)
            }.also {
                it.connect()
            }
        }

        var result = false
        udpClient?.let {
            it.send(broadcastMessage)
            result = true
        }
        return result
    }

    actual fun disconnect() {
        udpClient?.disconnect()
        udpClient = null
    }

    actual fun setReceiveCallback(callback: (message: String) -> Unit) {
        receiveCallback = callback
    }
}