package com.sentray.kmmprotocolmodule.cryptor.baseCryptor

import com.sentray.kmmprotocolmodule.cryptor.enumDefinition.CmdType
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.Cmd
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ProtocolKey
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.ProtocolVersion
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolParser.parserUtil.value
import korlibs.crypto.AES
import korlibs.crypto.Padding
import korlibs.crypto.encoding.Base64
import kotlinx.datetime.Clock
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.jsonObject
import kotlin.jvm.JvmStatic

internal class L2DataCryptor {
    @Serializable
    data class L2Cmd(
        val seq: Long,
        val protocolVersion: Int,
        val cmdType: Int,
        val l2Data: String,
    )

    private enum class Keys(val raw: String) {
        //IV密匙固定部分
        AES128_IV_FIX_PART(",4d!PK7#"),

        AES128_HTTP_KEY("3!FIAznT6+[E*.=W"),
        AES128_TCP_KEY("g,4PRK]Bw5k#b>XM"),
    }

    companion object {
        @ExperimentalSerializationApi
        @JvmStatic
        fun encodeTcp(l2DataRaw: String): String? {
            try {
                val jsonObject = Json.parseToJsonElement(l2DataRaw).jsonObject
                val seq = jsonObject.value<Long>(ProtocolKey.Seq.raw) ?: return null
                val cmd = kotlin.run {
                    jsonObject.value<String>(ProtocolKey.Cmd.raw)?.let {
                        Cmd.fromRaw(it)
                    }
                } ?: return null

                val protocolVersion = ProtocolVersion.V3.raw
                val cmdType = CmdType.getCmdType(cmd)
                val iv = ivKeyGen(seq.toString())

                val l2Data = encode(l2DataRaw, Keys.AES128_TCP_KEY, iv)
                val l2Cmd = L2Cmd(
                    seq,
                    protocolVersion,
                    cmdType.raw,
                    l2Data
                )
                return Json.encodeToString(l2Cmd)
            } catch (e: Exception) {
                e.printStackTrace()
                return null
            }
        }

        @JvmStatic
        fun encodeHttp(source: String): String {
            val timestamp = Clock.System.now().epochSeconds
            val timestampBase64 = Base64.encode(timestamp.toString().encodeToByteArray())

            val key = Keys.AES128_HTTP_KEY
            val iv = ivKeyGen(timestampBase64)

            return encode(source, key, iv)
        }

        @JvmStatic
        fun decodeTcp(jsonString: String): String? {
            try {
                val jsonObject = Json.parseToJsonElement(jsonString).jsonObject
                val seq = jsonObject.value<Long>(ProtocolKey.Seq.raw) ?: return null
                val iv = ivKeyGen(seq.toString())
                val key = Keys.AES128_TCP_KEY

                jsonObject.value<String>(ProtocolKey.L2Data.raw)?.let {
                    return decode(it, key, iv)
                } ?: kotlin.run {
                    return null
                }
            } catch (e: Exception) {
                e.printStackTrace()
                return null
            }
        }

        @JvmStatic
        fun decodeHttp(source: String): String? {
            try {
                val jsonObject = Json.parseToJsonElement(source).jsonObject
                val l2Data = jsonObject.value<String>(ProtocolKey.L2Data.raw) ?: return null
                val timestamp = jsonObject.value<String>(ProtocolKey.Timestamp.raw) ?: return null

                val key = Keys.AES128_HTTP_KEY
                val iv = ivKeyGen(timestamp)

                return decode(l2Data, key, iv)
            } catch (e: Exception) {
                e.printStackTrace()
                return null
            }
        }

        @JvmStatic
        private fun encode(source: String, key: Keys, iv: String): String {
            val aesEncodeData = AES.encryptAesCbc(
                source.encodeToByteArray(),
                key.raw.encodeToByteArray(),
                iv.encodeToByteArray(), Padding.PKCS7Padding
            )

            return Base64.encode(aesEncodeData)
        }

        @JvmStatic
        private fun decode(jsonString: String, key: Keys, iv: String): String? {
            val aesEncodeData = Base64.decode(jsonString)

            return AES.decryptAesCbc(
                aesEncodeData,
                key.raw.encodeToByteArray(),
                iv.encodeToByteArray(),
                Padding.PKCS7Padding
            ).decodeToString()
        }

        @JvmStatic
        private fun ivKeyGen(seq: String): String {
            var timestamp = seq
            if (timestamp.length < 8) {
                while (timestamp.length < 8) {
                    timestamp += "0"
                }
            } else {
                timestamp = timestamp.substring(timestamp.length - 8)
            }
            val seqBytes = timestamp.encodeToByteArray()
            val ivFixPartBytes: ByteArray = Keys.AES128_IV_FIX_PART.raw.encodeToByteArray()
            val ivKeyBytes = ByteArray(16)
            ivKeyBytes[0] = seqBytes[0]
            ivKeyBytes[1] = ivFixPartBytes[0]
            ivKeyBytes[2] = seqBytes[1]
            ivKeyBytes[3] = ivFixPartBytes[1]
            ivKeyBytes[4] = seqBytes[2]
            ivKeyBytes[5] = ivFixPartBytes[2]
            ivKeyBytes[6] = seqBytes[3]
            ivKeyBytes[7] = ivFixPartBytes[3]
            ivKeyBytes[8] = seqBytes[4]
            ivKeyBytes[9] = ivFixPartBytes[4]
            ivKeyBytes[10] = seqBytes[5]
            ivKeyBytes[11] = ivFixPartBytes[5]
            ivKeyBytes[12] = seqBytes[6]
            ivKeyBytes[13] = ivFixPartBytes[6]
            ivKeyBytes[14] = seqBytes[7]
            ivKeyBytes[15] = ivFixPartBytes[7]
            return ivKeyBytes.decodeToString()
        }
    }
}