package com.sentray.kmmprotocolmodule.tcpProtocol.protocolCreator

import com.sentray.kmmprotocolmodule.tcpProtocol.definition.deviceType.DeviceTypeSet
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.enumDefinition.*
import com.sentray.kmmprotocolmodule.tcpProtocol.protocolContent.paraContent.ParaStructure
import com.sentray.kmmprotocolmodule.utilityToolKit.HexToolKit
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.*
import kotlin.jvm.JvmStatic

internal class ControlCmdCreator {
    companion object {

        //设备开关控制命令
        @JvmStatic
        fun controlApplianceOnOff(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            action: Boolean,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexToHexString(applianceIndex)
                    val actionString = when (action) {
                        true -> maskString
                        false -> "0"
                    }
                    put(ProtocolKey.RlyMask.raw, maskString)
                    put(ProtocolKey.RlyAction.raw, actionString)
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun controlApplianceGroupOnOff(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndexList: List<Int>,
            action: Boolean,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexListToHexString(applianceIndexList)
                    val actionString = when (action) {
                        true -> maskString
                        false -> "0"
                    }
                    put(ProtocolKey.RlyMask.raw, maskString)
                    put(ProtocolKey.RlyAction.raw, actionString)
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun brightAdjust(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            brightPercent: Int,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexToHexString(applianceIndex)
                    put(ProtocolKey.RlyMask.raw, maskString)
                    put(ProtocolKey.RlyAction.raw, maskString)
                    put(ProtocolKey.BrightPercentMask.raw, maskString)
                    val brightPercentValid = when {
                        brightPercent < 0 -> 0
                        brightPercent > 100 -> 100
                        else -> brightPercent
                    }
                    put(ProtocolKey.BrightPercent.raw, brightPercentValid)
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun temperatureParaAdjust(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            brightPercent: Int,
            temperaturePercent: Int,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexToHexString(applianceIndex)
                    val para = ParaStructure()
                    para.setTemperatureLampBright(
                        when (brightPercent > 100) {
                            true -> 100.toULong()
                            false -> brightPercent.toULong()
                        }
                    )
                    para.setTemperatureLampTemp(
                        when (temperaturePercent > 100) {
                            true -> 100.toULong()
                            false -> temperaturePercent.toULong()
                        }
                    )
                    put(ProtocolKey.RlyMask.raw, maskString)
                    put(ProtocolKey.RlyAction.raw, maskString)
                    put(ProtocolKey.BrightPercentMask.raw, maskString)
                    put(ProtocolKey.BrightPercent.raw, para.getRawValue().toLong())
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun rgbwParaAdjust(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            para: ParaStructure,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val paraList: List<Int> = when (deviceTypeSet) {
                DeviceTypeSet.SonRGBLED -> listOf(
                    para.getColor(ParaStructure.RgbwType.Red),
                    para.getColor(ParaStructure.RgbwType.Green),
                    para.getColor(ParaStructure.RgbwType.Blue),
                    para.getColor(ParaStructure.RgbwType.Bright),
                )
                DeviceTypeSet.SonRGBWLED -> listOf(
                    para.getColor(ParaStructure.RgbwType.Red),
                    para.getColor(ParaStructure.RgbwType.Green),
                    para.getColor(ParaStructure.RgbwType.Blue),
                    para.getColor(ParaStructure.RgbwType.Bright),
                    para.getColor(ParaStructure.RgbwType.W),
                )
                else -> listOf()
            }
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexToHexString(applianceIndex)
                    put(ProtocolKey.RlyMask.raw, maskString)
                    put(ProtocolKey.RlyAction.raw, maskString)
                    put(ProtocolKey.ParaMask.raw, maskString)
                    putJsonArray(ProtocolKey.Para.raw) {
                        paraList.forEach { value ->
                            add(value)
                        }
                    }
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun irControl(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            irCmd: ParaStructure.IrCmd,
            irIndex: Int,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexToHexString(applianceIndex)
                    val para = ParaStructure()
                    para.setIRPara(ParaStructure.IrValueType.Cmd, irCmd.raw.toULong())
                    para.setIRPara(ParaStructure.IrValueType.Index, irIndex.toULong())
                    put(ProtocolKey.ParaMask.raw, maskString)
                    put(ProtocolKey.Para.raw, para.getRawValue().toLong())
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun curtainControl(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            curtainAction: ParaStructure.CurtainAction,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexToHexString(applianceIndex)
                    put(ProtocolKey.CurtainMask.raw, maskString)
                    put(ProtocolKey.CurtainAction.raw, curtainAction.raw)
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun curtainWithLocationPercentageControl(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            locationPercentage: Int,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexToHexString(applianceIndex)
                    put(ProtocolKey.ParaMask.raw, maskString)
                    var locationPercentageValid = when {
                        locationPercentage < 0 -> 0
                        locationPercentage > 100 -> 100
                        else -> locationPercentage
                    }

                    //由于中控协议定义问题，此处需作如下处理
                    if (locationPercentageValid == 100) {
                        locationPercentageValid = 0
                    } else if (locationPercentageValid == 0) {
                        locationPercentageValid = 100
                    }

                    put(ProtocolKey.Para.raw, locationPercentageValid)
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @JvmStatic
        fun curtainWithLocationActionControl(
            sn: String,
            deviceTypeSet: DeviceTypeSet,
            applianceIndex: Int,
            curtainAction: ParaStructure.CurtainWithLocAction,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val jsonObject = buildJsonObject {
                put(ProtocolKey.Sn.raw, sn)
                put(ProtocolKey.Seq.raw, seq)
                put(ProtocolKey.FactoryID.raw, factoryID.raw)
                put(ProtocolKey.Cmd.raw, Cmd.Set.raw)
                putJsonObject(deviceTypeSet.raw) {
                    val maskString = HexToolKit.indexToHexString(applianceIndex)
                    put(ProtocolKey.ParaMask.raw, maskString)
                    put(ProtocolKey.Para.raw, curtainAction.raw)
                }
            }

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(jsonObject)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }

        @Serializable
        data class ActivateTriggerPara(
            val sn: String,
            val seq: Int,
            val factoryID: Int,
            val cmd: String,
            val triggerIndex: Int,
        )

        @JvmStatic
        fun activateTrigger(
            sn: String,
            triggerIndex: Int,
            factoryID: FactoryID = FactoryID.WILINK,
            protocolVersion: ProtocolVersion = ProtocolVersion.V2,
            networkType: ENetworkType = ENetworkType.Cloud
        ): CommandData {
            val seq = Util.randomSeq()
            val para = ActivateTriggerPara(
                sn,
                seq,
                factoryID.raw,
                Cmd.DoTrigger.raw,
                triggerIndex
            )

            @OptIn(ExperimentalSerializationApi::class)
            val cmdRaw = Json.encodeToString(para)
            val cmdString = Util.encode(cmdRaw, protocolVersion, networkType)
            return CommandData(seq, cmdString)
        }
    }
}