package io.github.coho04.githubapi.entities;

import io.github.coho04.githubapi.Github;
import io.github.coho04.githubapi.bases.GHBase;
import org.json.JSONObject;

import java.util.List;

/**
 * This class represents a GitHub Vulnerability.
 * It provides methods for fetching data about the vulnerability such as its package, first patched version, vulnerable version range, and vulnerable functions.
 */
public class GHVulnerability extends GHBase {

    private GHPackage ghPackage;
    private final String firstPatchedVersion;
    private final String vulnerableVersionRange;
    private final List<String> vulnerableFunctions;

    /**
     * Constructs a new GHVulnerability instance with the provided JSON object.
     *
     * @param jsonObject the JSON object containing the vulnerability data
     */
    public GHVulnerability(Github github, JSONObject jsonObject) {
        if (jsonObject.has("package") && !jsonObject.isNull("package")) {
            this.ghPackage = new GHPackage(github, jsonObject.getJSONObject("package"));
        }
        this.firstPatchedVersion = getStringOrNull(jsonObject, "first_patched_version");
        this.vulnerableVersionRange = getStringOrNull(jsonObject, "vulnerable_version_range");
        this.vulnerableFunctions = getJSONArrayToStringList(jsonObject, "vulnerable_functions");
    }

    /**
     * Returns the package of the vulnerability.
     *
     * @return the package of the vulnerability
     */
    public GHPackage getGhPackage() {
        return ghPackage;
    }

    /**
     * Returns the first patched version of the vulnerability.
     *
     * @return the first patched version of the vulnerability
     */
    public String getFirstPatchedVersion() {
        return firstPatchedVersion;
    }

    /**
     * Returns the vulnerable version range of the vulnerability.
     *
     * @return the vulnerable version range of the vulnerability
     */
    public String getVulnerableVersionRange() {
        return vulnerableVersionRange;
    }

    /**
     * Returns the vulnerable functions of the vulnerability.
     *
     * @return the vulnerable functions of the vulnerability
     */
    public List<String> getVulnerableFunctions() {
        return vulnerableFunctions;
    }
}