package com.inmanlabs.commons.user;

import com.auth0.client.mgmt.filter.UserFilter;
import com.auth0.json.mgmt.users.User;
import com.auth0.json.mgmt.users.UsersPage;
import com.inmanlabs.commons.auth0.Auth0Client;
import com.inmanlabs.commons.web.error.FailedDependencyException;
import com.inmanlabs.commons.web.error.NotFoundException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Arrays;

@Service
public class UserService {

    private Logger logger = LoggerFactory.getLogger(UserService.class);

    private static String ANONYMOUS_USER_KEY = "anonymousUser";

    @Resource
    private UserRepository userRepository;

    @Resource
    private Auth0Client auth0Client;
//
//    public LocalUser safeSave(LocalUser user) {
//        logger.debug("Safe saving user...");
//        Optional<LocalUser> existingUser = this.userRepository.findByPhone(user.getPhone());
//        if (!existingUser.isPresent()) {
//            logger.debug("LocalUser did not exist, creating new one...");
//            return this.userRepository.save(user);
//        }
//
//        logger.debug("LocalUser exists already, updating existing one with legal fields...");
//        LocalUser updatedUser = existingUser.get();
//
//        updatedUser.setFirstName(user.getFirstName());
//        updatedUser.setLastName(user.getLastName());
//
//        return this.userRepository.save(updatedUser);
//    }

    public User currentUser() throws FailedDependencyException, NotFoundException {
        String currentUserId = SecurityContextHolder.getContext().getAuthentication().getName();
        if (currentUserId == null || StringUtils.equals(currentUserId, UserService.ANONYMOUS_USER_KEY)) {
            throw new NotFoundException("No user is currently logged in.");
        }
        return this.load(currentUserId);
    }

    public User load(String id) throws FailedDependencyException {
        return this.auth0Client.getUser(id);
    }

    public User findByPhone(String phoneNumber) throws FailedDependencyException {
        UserFilter filter = new UserFilter();
        filter.withQuery("phone_number:\"" + phoneNumber + "\"");
        UsersPage usersPage = this.auth0Client.list(filter);

        if (usersPage == null || usersPage.getLength() == null || usersPage.getLength() == 0) {
            return null;
        } else if (usersPage.getLength() > 1) {
            logger.error("We got more than one user when searching by phone number, this should not happen: " + Arrays.toString(usersPage.getItems().toArray()));
            throw new FailedDependencyException("There seems to be a problem with the auth0 filter");
        } else {
            return usersPage.getItems().get(0);
        }
    }

    public User put(String firstName, String lastName, String phoneNumber) throws FailedDependencyException {
        User user = this.findByPhone(phoneNumber);
        if (user == null) {
            return this.create(firstName, lastName, phoneNumber);
        } else {
            return this.update(user);
        }
    }

    public User update(User user) throws FailedDependencyException {
        User loadedUser = this.load(user.getId());

        // only copy over fields that are eligible for update. We don't allow the user to update any others.
        if (!StringUtils.equals(loadedUser.getFamilyName(), user.getFamilyName()) || !StringUtils.equals(user.getGivenName(), loadedUser.getGivenName())) {
            loadedUser.setFamilyName(user.getFamilyName());
            loadedUser.setGivenName(user.getGivenName());

            return this.auth0Client.update(user);
        }
        return loadedUser;
    }

    public User create(String firstName, String lastName, String phone) throws FailedDependencyException {
        User user = new User();
        user.setGivenName(firstName);
        user.setFamilyName(lastName);
        user.setPhoneNumber(phone);
        user.setPhoneVerified(true); // so they don't get an immediate sms. We verify upon entry to site.
        user.setConnection("sms");

        return this.auth0Client.create(user);
    }

//    public LocalUser create(String auth0Id, String phone, String email) {
//        LocalUser user = new LocalUser();
//        user.setAuthZeroId(auth0Id);
//        user.setPhone(phone);
//        user.setEmail(email);
//
//        return this.userRepository.save(user);
//    }
}
