package io.github.isagroup;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.logging.Logger;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.expression.spel.SpelEvaluationException;
import org.springframework.stereotype.Component;

import io.github.isagroup.exceptions.PricingPlanEvaluationException;
import io.github.isagroup.models.Feature;
import io.github.isagroup.models.FeatureStatus;
import io.github.isagroup.models.PlanContextManager;
import io.github.isagroup.models.PricingManager;
import io.github.isagroup.services.jwt.PricingJwtUtils;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.SignatureAlgorithm;

/**
 * Utility class that provides methods to generate and manage JWT that contains
 * the pricing plan evaluation context.
 */
@Component
public class PricingEvaluatorUtil {

    @Autowired
    private PricingJwtUtils jwtUtils;

    @Autowired
    private PricingContext pricingContext;

    public PricingEvaluatorUtil(PricingContext pricingContext) {
        this.pricingContext = pricingContext;
        this.jwtUtils = new PricingJwtUtils(pricingContext);
    }

    Logger logger = Logger.getLogger(PricingEvaluatorUtil.class.getName());

    /**
     * Generate a user authentication JWT that includes the pricing plan evaluation
     * context.
     * This token is generated by using the information provided by the configured
     * {@link PricingContext}
     * 
     * @return JWT that contains all the information
     */
    public String generateUserToken() {

        Map<String, Object> claims = new HashMap<>();
        
        claims.put("authorities", pricingContext.getUserAuthorities());

        String subject = "Default";

        if (pricingContext.getUserContext().containsKey("username")) {
            subject = (String) pricingContext.getUserContext().get("username");
        } else if (pricingContext.getUserContext().containsKey("user")) {
            subject = (String) pricingContext.getUserContext().get("user");
        }

        PlanContextManager planContextManager = new PlanContextManager();
        try{
            planContextManager.setUserContext(pricingContext.getUserContext());
            claims.put("userContext", planContextManager.getUserContext());
        }catch (Exception e){
            throw new PricingPlanEvaluationException("Error while retrieving user context! Please check your PricingContext.getUserContext() method");
        }

        if (!pricingContext.userAffectedByPricing()) {
            return Jwts.builder()
                .setClaims(claims)
                .setSubject(subject)
                .setIssuedAt(new Date(System.currentTimeMillis()))
                .setExpiration(new Date(System.currentTimeMillis() + pricingContext.getJwtExpiration()))
                .signWith(SignatureAlgorithm.HS512, pricingContext.getJwtSecret())
                .compact();
        }

        try{
            planContextManager.setPlanContext(pricingContext.getPlanContext());
        }catch (NullPointerException e){
            throw new PricingPlanEvaluationException("Error while retrieving plan context! Please check your configuration file or add a plan with the given name");
        }

        PricingManager pricingManager = pricingContext.getPricingManager();

        Map<String, Feature> features = pricingManager.getFeatures();

        Map<String, FeatureStatus> featureStatuses = computeFeatureStatuses(planContextManager, features);

        claims.put("features", featureStatuses);
        claims.put("planContext", planContextManager.getPlanContext());

        return Jwts.builder()
                .setClaims(claims)
                .setSubject(subject)
                .setIssuedAt(new Date(System.currentTimeMillis()))
                .setExpiration(new Date(System.currentTimeMillis() + pricingContext.getJwtExpiration()))
                .signWith(SignatureAlgorithm.HS512, pricingContext.getJwtSecret())
                .compact();
    }

    private Map<String, FeatureStatus> computeFeatureStatuses(PlanContextManager planContextManager,
            Map<String, Feature> features) {

        Map<String, FeatureStatus> featureStatuses = new HashMap<>();

        for (String featureName : features.keySet()) {

            FeatureStatus featureStatus = new FeatureStatus();
            Feature feature = features.get(featureName);

            String expression = features.get(featureName).getExpression();
            try{
                Boolean eval = FeatureStatus.computeFeatureEvaluation(expression, planContextManager)
                .orElseThrow(() -> new PricingPlanEvaluationException("Evaluation was null"));
                featureStatus.setEval(eval);
            }catch(SpelEvaluationException e){
                throw new PricingPlanEvaluationException("Error while evaluating the expression of the feature " + featureName + "! Please check the expression");
            }

            Optional<String> userContextKey = FeatureStatus.computeUserContextVariable(expression);

            if (!userContextKey.isPresent()) {
                featureStatus.setUsed(null);
                featureStatus.setLimit(null);
            } else {
                featureStatus.setUsed(planContextManager.getUserContext().get(userContextKey.get()));
                if(feature.getExpression().contains("usageLimits")){
                    String usageLimitName = feature.getExpression().split("usageLimits")[1].split("[',\"]")[2];
                    featureStatus.setLimit(((Map<String, Object>)planContextManager.getPlanContext().get("usageLimits")).get(usageLimitName));
                }else{
                    featureStatus.setLimit(((Map<String, Object>)planContextManager.getPlanContext().get("features")).get(featureName));
                }

            }

            featureStatuses.put(featureName, featureStatus);
        }
        return featureStatuses;

    }

    /**
     * Modifies the given JWT by changing the evaluation of the given feature by a
     * {@link String} expression that will be evaluated on the client side of the
     * application.
     * 
     * @param token      generated JWT returned by
     *                   {@link PricingEvaluatorUtil#generateUserToken()} method
     * @param featureId  the id of a feature that is defined inside the token body
     * @param expression the expression of the feature that will replace its
     *                   evaluation
     * @return Modified version of the provided JWT that contains the new expression
     *         in the "eval" attribute of the feature.
     */
    public String addExpressionToToken(String token, String featureId, String expression) {

        Map<String, Map<String, Object>> features = jwtUtils.getFeaturesFromJwtToken(token);
        String subject = jwtUtils.getSubjectFromJwtToken(token);

        try {
            Map<String, Object> feature = (Map<String, Object>) features.get(featureId);
            feature.put("eval", expression);
        } catch (Exception e) {
            logger.warning("Feature not found");
        }

        return buildJwtToken(features, subject);
    }

    private String buildJwtToken(Map<String, Map<String, Object>> features, String subject) {

        Map<String, Object> claims = new HashMap<>();

        claims.put("authorities", pricingContext.getUserAuthorities());
        claims.put("features", features);
        claims.put("userContext", pricingContext.getUserContext());
        claims.put("planContext", pricingContext.getPlanContext());

        return Jwts.builder()
                .setClaims(claims)
                .setSubject(subject)
                .setIssuedAt(new Date(System.currentTimeMillis()))
                .setExpiration(new Date(System.currentTimeMillis() + pricingContext.getJwtExpiration()))
                .signWith(SignatureAlgorithm.HS512, pricingContext.getJwtSecret())
                .compact();
    }

}
