package io.github.isagroup.services.jwt;

import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import io.github.isagroup.PricingContext;
import io.github.isagroup.PricingEvaluatorUtil;
import io.jsonwebtoken.ExpiredJwtException;
import io.jsonwebtoken.Jwts;
import io.jsonwebtoken.MalformedJwtException;
import io.jsonwebtoken.SignatureAlgorithm;
import io.jsonwebtoken.SignatureException;
import io.jsonwebtoken.UnsupportedJwtException;

/**
 * Utility class that provides methods to generate and manage JWT.
 */
@Component
public class PricingJwtUtils {

	@Autowired
	private PricingContext pricingContext;

	public PricingJwtUtils(PricingContext pricingContext) {
		this.pricingContext = pricingContext;
	}

	private static final Logger logger = LoggerFactory.getLogger(PricingJwtUtils.class);

	/**
	 * Extracts the subject from the given JWT.
	 * 
	 * @param token a JWT that contains a subject
	 * @return The subject of the JWT
	 */
	public String getSubjectFromJwtToken(String token) {
		return Jwts.parser().setSigningKey(pricingContext.getJwtSecret()).parseClaimsJws(token).getBody().getSubject();
	}

	/**
	 * Generates a JWT from the given username that does not contains pricing
	 * evaluation.
	 * 
	 * @param username The username to be used as the subject of the JWT
	 * @return The generated JWT
	 */
	public String generateTokenFromUsername(String username) {
		Map<String, Object> claims = new HashMap<>();
		claims.put("authorities", pricingContext.getUserAuthorities());
		return Jwts.builder().setClaims(claims).setSubject(username).setIssuedAt(new Date())
				.setExpiration(new Date((new Date()).getTime() + pricingContext.getJwtExpiration()))
				.signWith(SignatureAlgorithm.HS512, pricingContext.getJwtSecret()).compact();
	}

	/**
	 * Extracts the features from a JWT generated by
	 * {@link PricingEvaluatorUtil#generateUserToken()} method
	 * 
	 * @param token a JWT generated by
	 *              {@link PricingEvaluatorUtil#generateUserToken()} method
	 * @return A Map that contains the evaluation of all the features for the
	 *         current user located in the JWT body
	 */
	public Map<String, Map<String, Object>> getFeaturesFromJwtToken(String token) {
		return (Map<String, Map<String, Object>>) Jwts.parser().setSigningKey(pricingContext.getJwtSecret())
				.parseClaimsJws(token).getBody().get("features");
	}

	/**
	 * Extracts the plan context from a JWT generated by
	 * {@link PricingEvaluatorUtil#generateUserToken()} method
	 * 
	 * @param token a JWT generated by
	 *              {@link PricingEvaluatorUtil#generateUserToken()} method
	 * @return A Map that contains the plan context used in the evaluation of
	 *         features located in the JWT body
	 */
	public Map<String, Object> getPlanContextFromJwtToken(String token) {
		return (Map<String, Object>) Jwts.parser().setSigningKey(pricingContext.getJwtSecret()).parseClaimsJws(token)
				.getBody().get("planContext");
	}

	/**
	 * Extracts the user context from a JWT generated by
	 * {@link PricingEvaluatorUtil#generateUserToken()} method
	 * 
	 * @param token a JWT generated by
	 *              {@link PricingEvaluatorUtil#generateUserToken()} method
	 * @return A Map that contains the user context used in the evaluation of
	 *         features located in the JWT body
	 */
	public Map<String, Object> getUserContextFromJwtToken(String token) {
		return (Map<String, Object>) Jwts.parser().setSigningKey(pricingContext.getJwtSecret()).parseClaimsJws(token)
				.getBody().get("userContext");
	}

	/**
	 * Extracts the username from a JWT generated by
	 * {@link PricingEvaluatorUtil#generateUserToken()} method.
	 * It is the equivalent of getting the token's subject
	 * 
	 * @param token a JWT generated by
	 *              {@link PricingEvaluatorUtil#generateUserToken()} method
	 * @return The username of the user located in the JWT body
	 */
	public String getUserNameFromJwtToken(String token) {
		return Jwts.parser().setSigningKey(pricingContext.getJwtSecret()).parseClaimsJws(token).getBody().getSubject();
	}

	/**
	 * Validates a JWT
	 * It is the equivalent of getting the token's subject
	 * 
	 * @param authToken a JWT String
	 * @return true if the token is correct, false otherwise
	 */
	public boolean validateJwtToken(String authToken) {
		try {
			Jwts.parser().setSigningKey(pricingContext.getJwtSecret()).parseClaimsJws(authToken);
			return true;
		} catch (SignatureException e) {
			logger.error("Invalid JWT signature: {}", e.getMessage());
		} catch (MalformedJwtException e) {
			logger.error("Invalid JWT: {}", e.getMessage());
		} catch (ExpiredJwtException e) {
			logger.error("JWT is expired: {}", e.getMessage());
		} catch (UnsupportedJwtException e) {
			logger.error("JWT is unsupported: {}", e.getMessage());
		} catch (IllegalArgumentException e) {
			logger.error("JWT claims string is empty: {}", e.getMessage());
		}

		return false;
	}
}
