package io.github.koalaplot.core.bar

import androidx.compose.runtime.Composable
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.graphics.Color
import io.github.koalaplot.core.animation.StartAnimationUseCase
import io.github.koalaplot.core.style.KoalaPlotTheme
import io.github.koalaplot.core.xygraph.XYGraphScope

/**
 * Default implementation of a BarChartEntry.
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 */
public data class DefaultVerticalBarPlotEntry<X, Y>(
    public override val x: X,
    public override val y: BarPosition<Y>
) : VerticalBarPlotEntry<X, Y>

/**
 * An interface that defines a data element to be plotted on a Bar chart.
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 */
public interface VerticalBarPlotEntry<X, Y> {
    /**
     * X-axis value at which the bar should be plotted
     */
    public val x: X

    /**
     * The y-axis value for the bar.
     */
    public val y: BarPosition<Y>
}

/**
 * Convenience function for creating a VerticalBarPosition.
 */
public fun <Y> verticalBarPosition(yMin: Y, yMax: Y): BarPosition<Y> = DefaultBarPosition(yMin, yMax)

/**
 * Convenience function for creating a VerticalBarPlotEntry.
 */
public fun <X, Y> verticalBarPlotEntry(x: X, yMin: Y, yMax: Y): VerticalBarPlotEntry<X, Y> =
    DefaultVerticalBarPlotEntry(x, verticalBarPosition(yMin, yMax))

/**
 * Defines a Composable function used to emit a vertical bar.
 * The parameter series is the chart data series index.
 * The parameter index is the element index within the series.
 * The parameter value is the value of the element.
 */
public typealias VerticalBarComposable<E> = @Composable BarScope.(series: Int, index: Int, value: E) -> Unit

/**
 * Defines a Composable function used to emit a vertical bar for [VerticalBarPlotEntry] values.
 * Delegates to [VerticalBarComposable] with [VerticalBarPlotEntry] as type parameter.
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 */
public typealias DefaultVerticalBarComposable<X, Y> = VerticalBarComposable<VerticalBarPlotEntry<X, Y>>

/**
 * A VerticalBarPlot to be used in an XYGraph and that plots a single series of data points as vertical bars.
 *
 * @param X The type of the x-axis values
 * @param xData X-axis data points for where to plot the bars on the XYGraph. The size of xData and yData must match.
 * @param yData y-axis data points for each bar. Assumes each bar starts at 0.
 * @param bar Composable function to emit a bar for each data element, given the index of the point in the data and
 * the value of the data point.
 * @param barWidth The fraction of space between adjacent x-axis bars that may be used. Must be between 0 and 1,
 * defaults to 0.9.
 */
@Composable
public fun <X> XYGraphScope<X, Float>.VerticalBarPlot(
    xData: List<X>,
    yData: List<Float>,
    modifier: Modifier = Modifier,
    bar: DefaultVerticalBarComposable<X, Float>,
    barWidth: Float = 0.9f,
    startAnimationUseCase: StartAnimationUseCase =
        StartAnimationUseCase(
            executionType = StartAnimationUseCase.ExecutionType.Default,
            /* chart animation */
            KoalaPlotTheme.animationSpec,
        )
) {
    require(xData.size == yData.size) { "xData and yData must be the same size." }
    VerticalBarPlot(
        xData.mapIndexed { index, x ->
            DefaultVerticalBarPlotEntry(x, DefaultBarPosition(0f, yData[index]))
        },
        modifier,
        bar,
        barWidth,
        startAnimationUseCase
    )
}

/**
 * A VerticalBarPlot to be used in an XYGraph and that plots data points as vertical bars.
 *
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param E The type of the data element holding the values for each bar
 * @param data Data points for where to plot the bars on the XYGraph
 * @param bar Composable function to emit a bar for each data element, given the index of the point in the [data].
 * @param barWidth The fraction of space between adjacent x-axis bars that may be used. Must be between 0 and 1,
 * defaults to 0.9.
 */
@Composable
public fun <X, Y, E : VerticalBarPlotEntry<X, Y>> XYGraphScope<X, Y>.VerticalBarPlot(
    data: List<E>,
    modifier: Modifier = Modifier,
    bar: DefaultVerticalBarComposable<X, Y>,
    barWidth: Float = 0.9f,
    startAnimationUseCase: StartAnimationUseCase =
        StartAnimationUseCase(
            executionType = StartAnimationUseCase.ExecutionType.Default,
            /* chart animation */
            KoalaPlotTheme.animationSpec,
        )
) {
    val dataAdapter = remember(data) {
        VerticalEntryToGroupedEntryListAdapter(data)
    }

    // Delegate to the GroupedVerticalBarPlot - non-grouped is like grouped but with only 1 group per x-axis position
    GroupedVerticalBarPlot(
        dataAdapter,
        modifier = modifier,
        bar = { series, index, value ->
            bar(series, index, GroupedEntryToVerticalEntryAdapter(value))
        },
        maxBarGroupWidth = barWidth,
        startAnimationUseCase = startAnimationUseCase
    )
}

private class VerticalEntryToGroupedEntryListAdapter<X, Y>(
    val data: List<VerticalBarPlotEntry<X, Y>>
) : AbstractList<BarPlotGroupedPointEntry<X, Y>>() {
    override val size: Int
        get() = data.size

    override fun get(index: Int): BarPlotGroupedPointEntry<X, Y> {
        return VerticalEntryToGroupedEntryAdapter(data[index])
    }
}

private class VerticalEntryToGroupedEntryAdapter<X, Y>(val entry: VerticalBarPlotEntry<X, Y>) :
    BarPlotGroupedPointEntry<X, Y> {
    override val i: X = entry.x
    override val d: List<BarPosition<Y>>
        get() = object : AbstractList<BarPosition<Y>>() {
            override val size: Int = 1
            override fun get(index: Int): BarPosition<Y> = entry.y
        }
}

internal class GroupedEntryToVerticalEntryAdapter<X, Y>(
    private val entry: BarPlotGroupedPointEntry<X, Y>
) : VerticalBarPlotEntry<X, Y> {
    override val x: X
        get() = entry.i
    override val y: BarPosition<Y>
        get() = entry.d.first()
}

/**
 * Creates a Vertical Bar Plot.
 *
 * @param defaultBar A Composable to provide the bar if not specified on an individually added item.
 * @param barWidth The fraction of space between adjacent x-axis bars that may be used. Must be between 0 and 1,
 *  defaults to 0.9.
 * @param content A block which describes the content for the plot.
 */
@Composable
public fun <X, Y> XYGraphScope<X, Y>.VerticalBarPlot(
    defaultBar: DefaultVerticalBarComposable<X, Y> = verticalSolidBar(Color.Blue),
    modifier: Modifier = Modifier,
    barWidth: Float = 0.9f,
    startAnimationUseCase: StartAnimationUseCase =
        StartAnimationUseCase(
            executionType = StartAnimationUseCase.ExecutionType.Default,
            /* chart animation */
            KoalaPlotTheme.animationSpec,
        ),
    content: VerticalBarPlotScope<X, Y>.() -> Unit
) {
    val scope = remember(content, defaultBar) { VerticalBarPlotScopeImpl(defaultBar) }
    val data = remember(scope) {
        scope.content()
        scope.data.values.toList()
    }

    VerticalBarPlot(
        data.map { it.first },
        modifier,
        { series, index, value ->
            data[index].second.invoke(this, series, index, value)
        },
        barWidth,
        startAnimationUseCase
    )
}

/**
 * Scope item to allow adding items to a [VerticalBarPlot].
 */
public interface VerticalBarPlotScope<X, Y> {
    /**
     * Adds an item at the specified [x] axis coordinate, with vertical extent spanning from
     * [yMin] to [yMax]. An optional [bar] can be provided to customize the Composable used to
     * generate the bar for this specific item.
     */
    public fun item(x: X, yMin: Y, yMax: Y, bar: (DefaultVerticalBarComposable<X, Y>)? = null)
}

internal class VerticalBarPlotScopeImpl<X, Y>(private val defaultBar: DefaultVerticalBarComposable<X, Y>) :
    VerticalBarPlotScope<X, Y> {
    val data: MutableMap<X, Pair<VerticalBarPlotEntry<X, Y>, DefaultVerticalBarComposable<X, Y>>> =
        mutableMapOf()

    override fun item(x: X, yMin: Y, yMax: Y, bar: (DefaultVerticalBarComposable<X, Y>)?) {
        data[x] = Pair(verticalBarPlotEntry(x, yMin, yMax), bar ?: defaultBar)
    }
}
