@file:Suppress("TooManyFunctions") // TODO Remove once Deprecated functions removed

package io.github.koalaplot.core.line

import androidx.compose.animation.core.Animatable
import androidx.compose.animation.core.AnimationSpec
import androidx.compose.foundation.Canvas
import androidx.compose.foundation.layout.fillMaxSize
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.remember
import androidx.compose.ui.Modifier
import androidx.compose.ui.draw.drawWithContent
import androidx.compose.ui.geometry.Offset
import androidx.compose.ui.geometry.Size
import androidx.compose.ui.graphics.Path
import androidx.compose.ui.graphics.PathFillType
import androidx.compose.ui.graphics.StrokeCap
import androidx.compose.ui.graphics.drawscope.DrawScope
import androidx.compose.ui.graphics.drawscope.Fill
import androidx.compose.ui.graphics.drawscope.Stroke
import androidx.compose.ui.graphics.drawscope.clipRect
import androidx.compose.ui.layout.Layout
import androidx.compose.ui.layout.Measurable
import androidx.compose.ui.unit.Constraints
import io.github.koalaplot.core.style.AreaStyle
import io.github.koalaplot.core.style.KoalaPlotTheme
import io.github.koalaplot.core.style.LineStyle
import io.github.koalaplot.core.util.HoverableElementAreaScope
import io.github.koalaplot.core.util.lineTo
import io.github.koalaplot.core.util.moveTo
import io.github.koalaplot.core.xygraph.Point
import io.github.koalaplot.core.xygraph.XYGraphScope
import kotlin.apply
import kotlin.math.min

internal const val DefaultTau = 0.5f

/**
 * A line plot that draws data as points and lines on an XYGraph.
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param data Data series to plot.
 * @param lineStyle Style to use for the line that connects the data points. If null, no line is drawn.
 * @param symbol Composable for the symbol to be shown at each data point.
 * @param modifier Modifier for the plot.
 */
@Composable
public fun <X, Y> XYGraphScope<X, Y>.LinePlot2(
    data: List<Point<X, Y>>,
    modifier: Modifier = Modifier,
    lineStyle: LineStyle? = null,
    symbol: (@Composable (Point<X, Y>) -> Unit)? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
) {
    LinePlot(data, modifier, lineStyle, { symbol?.invoke(it) }, animationSpec)
}

/**
 * A line plot that draws data as points and lines on an XYGraph.
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param data Data series to plot.
 * @param lineStyle Style to use for the line that connects the data points. If null, no line is drawn.
 * @param symbol Composable for the symbol to be shown at each data point.
 * @param modifier Modifier for the plot.
 */
@Deprecated("Use LinePlot2 instead", replaceWith = ReplaceWith("LinePlot2"))
@Composable
public fun <X, Y> XYGraphScope<X, Y>.LinePlot(
    data: List<Point<X, Y>>,
    modifier: Modifier = Modifier,
    lineStyle: LineStyle? = null,
    symbol: (@Composable HoverableElementAreaScope.(Point<X, Y>) -> Unit)? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
) {
    if (data.isEmpty()) return

    GeneralLinePlot(
        data,
        modifier,
        lineStyle,
        symbol,
        null,
        null,
        animationSpec,
    ) { points: List<Point<X, Y>>, size: Size ->
        moveTo(scale(points[0], size))
        for (index in 1..points.lastIndex) {
            lineTo(scale(points[index], size))
        }
    }
}

/**
 * A line plot that draws a smooth, curved line that passes through each data point.
 *
 * This plot is ideal for representing continuous data sets where a visually smooth path is desired,
 * such as in signal processing or natural phenomena graphs. It differs from a standard [LinePlot2] by using
 * cubic Bézier curves between points instead of straight lines.
 *
 * @param X The type of the x-axis values.
 * @param Y The type of the y-axis values.
 * @param data The series of `Point`s to be plotted.
 * @param control A [CubicBezierControlPointCalculator] that defines the curve's shape between points.
 * Defaults to a Catmull-Rom implementation.
 * @param lineStyle Style for the line connecting data points. If null, no line is drawn.
 * @param symbol An optional composable used to render a symbol at each data point.
 * @param animationSpec The [AnimationSpec] to use for animating the plot.
 * @param modifier Modifier for the plot.
 */
@Composable
public fun <X, Y> XYGraphScope<X, Y>.CubicBezierLinePlot(
    data: List<Point<X, Y>>,
    modifier: Modifier = Modifier,
    control: CubicBezierControlPointCalculator = catmullRomControlPoints(DefaultTau),
    lineStyle: LineStyle? = null,
    symbol: (@Composable (Point<X, Y>) -> Unit)? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
) {
    if (data.isEmpty()) return

    val cubicBezierGenerator = cubicBezierDrawer(control)

    GeneralLinePlot(
        data,
        modifier,
        lineStyle,
        { symbol?.invoke(it) },
        null,
        null,
        animationSpec,
        cubicBezierGenerator,
    )
}

/**
 * An XY Chart that draws series as points and stairsteps between points.
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param data Data series to plot.
 * @param lineStyle Style to use for the line that connects the data points.
 * @param areaStyle Style to use for filling the area between the line and the 0-cross of the y-axis, or the
 *  y-axis value closest to 0 if the axis does not include 0. If null, no area will be drawn.
 *  [lineStyle] must also be non-null for the area to be drawn.
 * each point having the same x-axis value.
 * @param areaBaseline Baseline location for the area. Must be not be null if areaStyle and lineStyle are also not null.
 * If [areaBaseline] is an [AreaBaseline.ArbitraryLine] then the size of the line data must be equal to that of
 * [data], and their x-axis values must match.
 * @param symbol Composable for the symbol to be shown at each data point.
 * @param modifier Modifier for the chart.
 */
@Composable
public fun <X, Y> XYGraphScope<X, Y>.StairstepPlot2(
    data: List<Point<X, Y>>,
    lineStyle: LineStyle,
    modifier: Modifier = Modifier,
    symbol: (@Composable (Point<X, Y>) -> Unit)? = null,
    areaStyle: AreaStyle? = null,
    areaBaseline: AreaBaseline<X, Y>? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
) {
    StairstepPlot(data, lineStyle, modifier, { symbol?.invoke(it) }, areaStyle, areaBaseline, animationSpec)
}

/**
 * An XY Chart that draws series as points and stairsteps between points.
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param data Data series to plot.
 * @param lineStyle Style to use for the line that connects the data points.
 * @param areaStyle Style to use for filling the area between the line and the 0-cross of the y-axis, or the
 *  y-axis value closest to 0 if the axis does not include 0. If null, no area will be drawn.
 *  [lineStyle] must also be non-null for the area to be drawn.
 * each point having the same x-axis value.
 * @param areaBaseline Baseline location for the area. Must be not be null if areaStyle and lineStyle are also not null.
 * If [areaBaseline] is an [AreaBaseline.ArbitraryLine] then the size of the line data must be equal to that of
 * [data], and their x-axis values must match.
 * @param symbol Composable for the symbol to be shown at each data point.
 * @param modifier Modifier for the chart.
 */
@Deprecated("Use StairstepPlot2 instead", replaceWith = ReplaceWith("StairstepPlot2"))
@Composable
public fun <X, Y> XYGraphScope<X, Y>.StairstepPlot(
    data: List<Point<X, Y>>,
    lineStyle: LineStyle,
    modifier: Modifier = Modifier,
    symbol: (@Composable HoverableElementAreaScope.(Point<X, Y>) -> Unit)? = null,
    areaStyle: AreaStyle? = null,
    areaBaseline: AreaBaseline<X, Y>? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
) {
    if (data.isEmpty()) return

    if (areaStyle != null) {
        require(areaBaseline != null) { "areaBaseline must be provided for area charts" }
        if (areaBaseline is AreaBaseline.ArbitraryLine) {
            require(areaBaseline.values.size == data.size) {
                "baseline values must be the same size as the data"
            }
        }
    }

    GeneralLinePlot(
        data,
        modifier,
        lineStyle,
        symbol,
        areaStyle,
        areaBaseline,
        animationSpec,
    ) { points: List<Point<X, Y>>, size: Size ->
        var lastPoint = points[0]
        var scaledLastPoint = scale(lastPoint, size)

        moveTo(scaledLastPoint)
        for (index in 1..points.lastIndex) {
            val midPoint = scale(Point(x = points[index].x, y = lastPoint.y), size)
            lineTo(midPoint)
            lastPoint = points[index]
            scaledLastPoint = scale(lastPoint, size)
            lineTo(scaledLastPoint)
        }
    }
}

/**
 * A [StairstepPlot] that differentiate [lineStyle] & [areaBaseline] at each [Y]-values based on [levelLineStyle].
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param data Data series to plot.
 * @param lineStyle Style to use for the line that connects the data points.
 * @param levelLineStyle Style to use for emphasizing the y-axis values. (Used for line that connects same-level
 *  data points, data that have same value ([Y]) should have the same style).
 * @param cap Choose the [StrokeCap] used for level lines ending.
 * @param areaStyle Style to use for filling the area between the line and the 0-cross of the y-axis, or the
 *  y-axis value closest to 0 if the axis does not include 0. If null, no area will be drawn.
 *  [lineStyle] must also be non-null for the area to be drawn.
 * each point having the same x-axis value.
 * @param areaBaseline Baseline location for the area. Must be not be null if areaStyle and lineStyle are also not null.
 * If [areaBaseline] is an [AreaBaseline.ArbitraryLine] then the size of the line data must be equal to that of
 * [data], and their x-axis values must match.
 * @param symbol Composable for the symbol to be shown at each data point.
 * @param modifier Modifier for the chart.
 */
@Composable
@Suppress("ktlint:compose:param-order-check")
public fun <X, Y> XYGraphScope<X, Y>.StairstepPlot3(
    data: List<Point<X, Y>>,
    lineStyle: LineStyle,
    levelLineStyle: (Y) -> LineStyle,
    modifier: Modifier = Modifier,
    cap: StrokeCap = StrokeCap.Square,
    symbol: @Composable ((Point<X, Y>) -> Unit)? = null,
    areaStyle: ((Y) -> AreaStyle)? = null,
    areaBaseline: AreaBaseline<X, Y>? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
) {
    StairstepPlot(
        data,
        lineStyle,
        levelLineStyle,
        cap,
        modifier,
        { symbol?.invoke(it) },
        areaStyle,
        areaBaseline,
        animationSpec,
    )
}

/**
 * A [StairstepPlot] that differentiate [lineStyle] & [areaBaseline] at each [Y]-values based on [levelLineStyle].
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param data Data series to plot.
 * @param lineStyle Style to use for the line that connects the data points.
 * @param levelLineStyle Style to use for emphasizing the y-axis values. (Used for line that connects same-level
 *  data points, data that have same value ([Y]) should have the same style).
 * @param cap Choose the [StrokeCap] used for level lines ending.
 * @param areaStyle Style to use for filling the area between the line and the 0-cross of the y-axis, or the
 *  y-axis value closest to 0 if the axis does not include 0. If null, no area will be drawn.
 *  [lineStyle] must also be non-null for the area to be drawn.
 * each point having the same x-axis value.
 * @param areaBaseline Baseline location for the area. Must be not be null if areaStyle and lineStyle are also not null.
 * If [areaBaseline] is an [AreaBaseline.ArbitraryLine] then the size of the line data must be equal to that of
 * [data], and their x-axis values must match.
 * @param symbol Composable for the symbol to be shown at each data point.
 * @param modifier Modifier for the chart.
 */
@Composable
@Suppress("ktlint:compose:param-order-check")
@Deprecated("Use StairstepPlot3 instead", replaceWith = ReplaceWith("StairstepPlot3"))
public fun <X, Y> XYGraphScope<X, Y>.StairstepPlot2(
    data: List<Point<X, Y>>,
    lineStyle: LineStyle,
    levelLineStyle: (Y) -> LineStyle,
    cap: StrokeCap = StrokeCap.Square,
    modifier: Modifier = Modifier,
    symbol: @Composable ((Point<X, Y>) -> Unit)? = null,
    areaStyle: ((Y) -> AreaStyle)? = null,
    areaBaseline: AreaBaseline<X, Y>? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
) {
    StairstepPlot(
        data,
        lineStyle,
        levelLineStyle,
        cap,
        modifier,
        { symbol?.invoke(it) },
        areaStyle,
        areaBaseline,
        animationSpec,
    )
}

/**
 * A [StairstepPlot] that differentiate [lineStyle] & [areaBaseline] at each [Y]-values based on [levelLineStyle].
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param data Data series to plot.
 * @param lineStyle Style to use for the line that connects the data points.
 * @param levelLineStyle Style to use for emphasizing the y-axis values. (Used for line that connects same-level
 *  data points, data that have same value ([Y]) should have the same style).
 * @param cap Choose the [StrokeCap] used for level lines ending.
 * @param areaStyle Style to use for filling the area between the line and the 0-cross of the y-axis, or the
 *  y-axis value closest to 0 if the axis does not include 0. If null, no area will be drawn.
 *  [lineStyle] must also be non-null for the area to be drawn.
 * each point having the same x-axis value.
 * @param areaBaseline Baseline location for the area. Must be not be null if areaStyle and lineStyle are also not null.
 * If [areaBaseline] is an [AreaBaseline.ArbitraryLine] then the size of the line data must be equal to that of
 * [data], and their x-axis values must match.
 * @param symbol Composable for the symbol to be shown at each data point.
 * @param modifier Modifier for the chart.
 */
@Deprecated("Use StairstepPlot2 instead", replaceWith = ReplaceWith("StairstepPlot2"))
@Composable
@Suppress("LongMethod", "CyclomaticComplexMethod")
public fun <X, Y> XYGraphScope<X, Y>.StairstepPlot(
    data: List<Point<X, Y>>,
    lineStyle: LineStyle,
    levelLineStyle: (Y) -> LineStyle,
    cap: StrokeCap = StrokeCap.Square,
    modifier: Modifier = Modifier,
    symbol: @Composable (HoverableElementAreaScope.(Point<X, Y>) -> Unit)? = null,
    areaStyle: ((Y) -> AreaStyle)? = null,
    areaBaseline: AreaBaseline<X, Y>? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
) {
    if (data.isEmpty()) return

    if (areaStyle != null) {
        require(areaBaseline != null) { "areaBaseline must be provided for area charts" }
        require(areaBaseline !is AreaBaseline.CubicBezierLine) {
            "Cubic Bezier baselines are not supported for StairstepPlot"
        }
        if (areaBaseline is AreaBaseline.ArbitraryLine) {
            require(areaBaseline.values.size == data.size) {
                "baseline values must be the same size as the data"
            }
        }
    }

    // Modified version of [GeneralLinePlot].

    // Animation scale factor
    val beta = remember { Animatable(0f) }
    LaunchedEffect(null) { beta.animateTo(1f, animationSpec = animationSpec) }

    Layout(
        modifier = modifier.drawWithContent {
            clipRect(right = size.width * beta.value) { (this@drawWithContent).drawContent() }
        },
        content = {
            Canvas(modifier = Modifier.fillMaxSize()) {
                data class OffsetPoint(
                    val offset: Offset,
                    val point: Point<X, Y>,
                )

                // Order of executing: [onFirstPoint] -> [onMidPoint] -> [onNextPoint] -> [onMidPoint] ...,
                // so `nextPoint` of [onNextPoint] will becomes `lastPoint` of [onMidPoint].
                fun scaledPointsVisitor(
                    points: List<Point<X, Y>>,
                    onFirstPoint: (OffsetPoint) -> Unit = { _ -> },
                    onMidPoint: (lastPoint: OffsetPoint, midPoint: OffsetPoint) -> Unit = { _, _ -> },
                    onNextPoint: (midPoint: OffsetPoint, nextPoint: OffsetPoint) -> Unit = { _, _ -> },
                ) {
                    var lastPoint = points[0]
                    var scaledLastPoint = scale(lastPoint, size)
                    var offsetLastPoint = OffsetPoint(scaledLastPoint, lastPoint)

                    onFirstPoint(offsetLastPoint)
                    for (index in 1..points.lastIndex) {
                        val scaledMidPoint = scale(Point(x = points[index].x, y = lastPoint.y), size)
                        val midPoint = OffsetPoint(scaledMidPoint, lastPoint)
                        onMidPoint(offsetLastPoint, midPoint)
                        lastPoint = points[index]
                        scaledLastPoint = scale(lastPoint, size)
                        offsetLastPoint = OffsetPoint(scaledLastPoint, lastPoint)
                        onNextPoint(midPoint, offsetLastPoint)
                    }
                }
                if (areaBaseline != null && areaStyle != null) {
                    var i = 0
                    var lastPoint: OffsetPoint? = null
                    scaledPointsVisitor(
                        data,
                        onMidPoint = { lp, _ -> lastPoint = lp },
                        onNextPoint = { midPoint, nextPoint ->
                            fillRectangle(
                                leftTop = lastPoint!!.offset,
                                rightBottom = scale(
                                    Point(
                                        nextPoint.point.x,
                                        when (areaBaseline) {
                                            is AreaBaseline.HorizontalLine -> areaBaseline.value

                                            is AreaBaseline.ConstantLine -> areaBaseline.value

                                            is AreaBaseline.ArbitraryLine -> areaBaseline.values[i].y

                                            is AreaBaseline.CubicBezierLine -> error(
                                                "Cubic Bezier baselines are not supported for StairstepPlot",
                                            )
                                        },
                                    ),
                                    size,
                                ),
                                areaStyle = areaStyle(midPoint.point.y),
                            )
                            i++
                        },
                    )
                }

                // draw vertical lines using lineStyle
                scaledPointsVisitor(
                    data,
                    onNextPoint = { midPoint, p ->
                        with(lineStyle) {
                            drawLine(
                                brush,
                                midPoint.offset,
                                p.offset,
                                strokeWidth.toPx(),
                                Stroke.DefaultCap,
                                pathEffect,
                                alpha,
                                colorFilter,
                                blendMode,
                            )
                        }
                    },
                )
                // draw horizontal lines using levelLineStyle()
                scaledPointsVisitor(
                    data,
                    onMidPoint = { lastPoint, p ->
                        with(levelLineStyle(p.point.y)) {
                            drawLine(brush, lastPoint.offset, p.offset, strokeWidth.toPx(), cap, pathEffect, alpha)
                        }
                    },
                )
            }
            Symbols(data, symbol)
        },
    ) { measurables: List<Measurable>, constraints: Constraints ->
        layout(constraints.maxWidth, constraints.maxHeight) {
            measurables.forEach {
                it.measure(constraints).place(0, 0)
            }
        }
    }
}

/**
 * @param X The type of the x-axis values
 * @param Y The type of the y-axis values
 * @param data Data series to plot.
 * @param lineStyle Style to use for the line that connects the data points. If null, no line is drawn.
 * @param symbol Composable for the symbol to be shown at each data point.
 * @param modifier Modifier for the chart.
 */
@Composable
internal fun <X, Y> XYGraphScope<X, Y>.GeneralLinePlot(
    data: List<Point<X, Y>>,
    modifier: Modifier = Modifier,
    lineStyle: LineStyle? = null,
    symbol: (@Composable HoverableElementAreaScope.(Point<X, Y>) -> Unit)? = null,
    areaStyle: AreaStyle? = null,
    areaBaseline: AreaBaseline<X, Y>? = null,
    animationSpec: AnimationSpec<Float> = KoalaPlotTheme.animationSpec,
    drawConnectorLine: LineDrawer<X, Y>,
) {
    if (data.isEmpty()) return

    // Animation scale factor
    val beta = remember { Animatable(0f) }
    LaunchedEffect(null) { beta.animateTo(1f, animationSpec = animationSpec) }

    Layout(
        modifier = modifier.drawWithContent {
            clipRect(right = size.width * beta.value) { (this@drawWithContent).drawContent() }
        },
        content = {
            Canvas(modifier = Modifier.fillMaxSize()) {
                val mainLinePath = Path().apply {
                    with(drawConnectorLine) { draw(data, size) }
                }

                if (areaBaseline != null && areaStyle != null) {
                    val areaPath = generateArea(areaBaseline, data, mainLinePath, size, drawConnectorLine)
                    drawPath(
                        areaPath,
                        brush = areaStyle.brush,
                        alpha = areaStyle.alpha,
                        style = Fill,
                        colorFilter = areaStyle.colorFilter,
                        blendMode = areaStyle.blendMode,
                    )
                }

                lineStyle?.let {
                    drawPath(
                        mainLinePath,
                        brush = lineStyle.brush,
                        alpha = lineStyle.alpha,
                        style = Stroke(lineStyle.strokeWidth.toPx(), pathEffect = lineStyle.pathEffect),
                        colorFilter = lineStyle.colorFilter,
                        blendMode = lineStyle.blendMode,
                    )
                }
            }
            Symbols(data, symbol)
        },
    ) { measurables: List<Measurable>, constraints: Constraints ->
        layout(constraints.maxWidth, constraints.maxHeight) {
            measurables.forEach {
                it.measure(constraints).place(0, 0)
            }
        }
    }
}

private fun <X, Y> XYGraphScope<X, Y>.generateArea(
    areaBaseline: AreaBaseline<X, Y>,
    data: List<Point<X, Y>>,
    mainLinePath: Path,
    size: Size,
    drawConnectorLine: LineDrawer<X, Y>,
): Path = Path().apply {
    fillType = PathFillType.EvenOdd
    addPath(mainLinePath)

    when (areaBaseline) {
        is AreaBaseline.ArbitraryLine -> {
            // right edge of fill area
            lineTo(scale(areaBaseline.values.last(), size))

            // draw baseline
            with(drawConnectorLine) {
                draw(areaBaseline.values.reversed(), size)
            }
        }

        is AreaBaseline.CubicBezierLine -> {
            // right edge
            lineTo(scale(areaBaseline.values.last(), size))

            // draw baseline
            with(drawConnectorLine) {
                draw(areaBaseline.values.reversed(), size)
            }
        }

        is AreaBaseline.ConstantLine -> {
            // right edge
            lineTo(scale(Point(data.last().x, areaBaseline.value), size))

            // baseline
            lineTo(scale(Point(data.first().x, areaBaseline.value), size))
        }

        is AreaBaseline.HorizontalLine -> {
            // right edge
            lineTo(scale(Point(data.last().x, areaBaseline.value), size))

            // baseline
            lineTo(scale(Point(data.first().x, areaBaseline.value), size))
        }
    }

    // draw left edge of fill area
    lineTo(scale(data.first(), size))

    close()
}

private fun DrawScope.fillRectangle(
    leftTop: Offset,
    rightBottom: Offset,
    areaStyle: AreaStyle,
) {
    drawRect(
        brush = areaStyle.brush,
        topLeft = leftTop,
        size = (rightBottom - leftTop).run { Size(x, y) },
        alpha = areaStyle.alpha,
        style = Fill,
        colorFilter = areaStyle.colorFilter,
        blendMode = areaStyle.blendMode,
    )
}

@Composable
private fun <X, Y, P : Point<X, Y>> XYGraphScope<X, Y>.Symbols(
    data: List<P>,
    symbol: (@Composable HoverableElementAreaScope.(P) -> Unit)? = null,
) {
    if (symbol != null) {
        Layout(
            modifier = Modifier.fillMaxSize(),
            content = {
                data.indices.forEach {
                    symbol.invoke(this, data[it])
                }
            },
        ) { measurables: List<Measurable>, constraints: Constraints ->
            val size = Size(constraints.maxWidth.toFloat(), constraints.maxHeight.toFloat())

            layout(constraints.maxWidth, constraints.maxHeight) {
                for (index in 0 until min(data.size, measurables.size)) {
                    val p = measurables[index].measure(constraints.copy(minWidth = 0, minHeight = 0))
                    var position = scale(data[index], size)
                    position -= Offset(p.width / 2f, p.height / 2f)
                    p.place(position.x.toInt(), position.y.toInt())
                }
            }
        }
    }
}

/**
 * An interface for drawing a connector line on a [Path].
 * This is used by plots to define how a series of points are translated into a drawable path.
 */
internal fun interface LineDrawer<X, Y> {
    /**
     * Draws the connector line for the given [data] points.
     * @param data The list of data points to connect.
     * @param size The available size of the drawing area.
     */
    fun Path.draw(
        data: List<Point<X, Y>>,
        size: Size,
    )
}

/**
 * An interface for computing the two control points of a cubic Bézier curve segment.
 * The curve segment connects the current and next data points.
 */
public fun interface CubicBezierControlPointCalculator {
    /**
     * Computes the two control points for a cubic Bézier curve segment.
     *
     * The first returned [Offset] corresponds to control point 1 (x1, y1) and the second to
     * control point 2 (x2, y2), as defined in [Path.cubicTo].
     * https://developer.android.com/reference/kotlin/androidx/compose/ui/graphics/Path#cubicTo(kotlin.Float,kotlin.Float,kotlin.Float,kotlin.Float,kotlin.Float,kotlin.Float)
     *
     * @param previous The data point before the [current] point in the series.
     * @param current The starting point of the Bézier curve segment.
     * @param next The ending point of the Bézier curve segment.
     * @param subsequent The data point after the [next] point in the series.
     * @return A [Pair] containing the two control points for the curve segment.
     */
    public fun compute(
        previous: Offset,
        current: Offset,
        next: Offset,
        subsequent: Offset,
    ): Pair<Offset, Offset>
}

/**
 * Creates a [CubicBezierControlPointCalculator] that generates control points for a Catmull-Rom spline,
 * resulting in a smooth curve that passes through all data points.
 *
 * This implementation uses the position of neighboring points to calculate tangents, ensuring a
 * continuous curve.
 *
 * @param tau A value, typically between 0.0f and 1.0f, that controls the "curviness" of the spline.
 * A value of 0.0f will produce straight lines. A common default is 0.5f. The 'tau' parameter from
 * Catmull-Rom theory is divided by 3 internally to adapt it for cubic Bézier usage.
 * @return A [CubicBezierControlPointCalculator] for use in [CubicBezierLinePlot].
 * @see <a href="https://en.wikipedia.org/wiki/Catmull%E2%80%93Rom_spline">Catmull-Rom spline (Wikipedia)</a>
 */
@Suppress("MagicNumber")
public fun catmullRomControlPoints(tau: Float): CubicBezierControlPointCalculator =
    CubicBezierControlPointCalculator { previous, current, next, subsequent ->
        val cp1 = current + Offset(next.x - previous.x, next.y - previous.y) * tau / 3f
        val cp2 = next - Offset(subsequent.x - current.x, subsequent.y - current.y) * tau / 3f
        cp1 to cp2
    }

/**
 * Creates a [CubicBezierControlPointCalculator] that generates control points for a horizontal Bézier curve.
 *
 * This calculator places the control points halfway horizontally between the current and next data points,
 * at the same y-level as the current and next points respectively. The resulting curve leaves the first
 * point horizontally and arrives at the second point horizontally.
 *
 * @return A [CubicBezierControlPointCalculator] for use in [CubicBezierLinePlot].
 */
public fun horizontalBezierControlPoints(): CubicBezierControlPointCalculator = CubicBezierControlPointCalculator { _, current, next, _ ->
    val dx = (current.x + next.x) / 2
    Offset(dx, current.y) to Offset(dx, next.y)
}

internal fun <X, Y> XYGraphScope<X, Y>.cubicBezierDrawer(control: CubicBezierControlPointCalculator): LineDrawer<X, Y> =
    LineDrawer { points, size ->
        moveTo(scale(points[0], size))
        for (index in 1..points.lastIndex) {
            val (cp1, cp2) = control.compute(
                scale(points[(index - 2).coerceAtLeast(0)], size),
                scale(points[index - 1], size),
                scale(points[index], size),
                scale(points[(index + 1).coerceAtMost(points.lastIndex)], size),
            )
            val p3 = scale(points[index], size)
            cubicTo(cp1.x, cp1.y, cp2.x, cp2.y, p3.x, p3.y)
        }
    }

internal fun <X, Y> XYGraphScope<X, Y>.lineDrawer(): LineDrawer<X, Y> = LineDrawer { points, size ->
    moveTo(scale(points[0], size))
    for (index in 1..points.lastIndex) {
        lineTo(scale(points[index], size))
    }
}
