package com.liecoder.framework.base

import android.content.Intent
import android.content.pm.PackageManager
import android.net.Uri
import android.os.Bundle
import android.provider.Settings
import android.view.Gravity
import androidx.activity.result.ActivityResultCallback
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.appcompat.app.AlertDialog
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import com.liecoder.framework.model.Permission
import com.liecoder.framework.toast.TipsToast.Errors

/**
 * 基础权限活动类，提供了一套标准的方式来处理运行时权限请求。
 *
 * 此类继承自 AppCompatActivity 并实现了 IPermissionService 接口和 ActivityResultCallback，
 * 用于简化权限请求流程并提供统一的权限管理策略。
 *
 * @author liecoder
 * @date 2024/6/21 周五
 * @version 1.0
 */
abstract class BasePermissionActivity : AppCompatActivity(), IPermissionService,
    ActivityResultCallback<Boolean> {


    // 日志标签，使用当前类的简单名称。
    protected val TAG = this::class.java.simpleName


    // 用于请求权限的 ActivityResultLauncher，延迟初始化。
    private lateinit var requestPermissionLauncher: ActivityResultLauncher<String>


    // 权限授予时的回调。
    private var onPermissionGranted: (() -> Unit)? = null


    // 权限拒绝时的回调。
    private var onPermissionDenied: (() -> Unit)? = null


    // 权限请求说明对话框。
    private var permissionExplanationDialog: AlertDialog? = null


    /**
     * 活动创建时的回调方法。
     *
     * 在这里初始化 requestPermissionLauncher。
     * @param savedInstanceState 保存的实例状态。
     */
    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        requestPermissionLauncher = registerPermissionLauncher()
    }


    /**
     * 注册权限请求的 ActivityResultLauncher。
     *
     * @return 配置好的 ActivityResultLauncher 实例。
     */
    override fun registerPermissionLauncher(): ActivityResultLauncher<String> {
        return registerForActivityResult(
            ActivityResultContracts.RequestPermission(), this
        )
    }


    /**
     * 处理权限请求的结果。
     *
     * @param isGranted 权限是否被授予。
     */
    final override fun onActivityResult(isGranted: Boolean?) {
        if (isGranted == true) {
            // 权限被授予，调用权限授予的回调。
            onPermissionGranted?.invoke()
        } else {
            // 权限被拒绝，调用权限拒绝的回调。
            onPermissionDenied?.invoke()
        }
        // 无论结果如何，都关闭权限说明对话框。
        permissionExplanationDialog?.dismiss()
    }


    /**
     * 处理权限请求逻辑。
     *
     * @param permission 请求的权限。
     * @param onPermissionGranted 权限授予时调用的回调。
     * @param onPermissionDenied 权限拒绝时调用的回调。
     */
    override fun withPermission(
        permission: Permission, onPermissionDenied: (() -> Unit)?, onPermissionGranted: () -> Unit
    ) {
        // 设置权限请求回调。
        this.onPermissionGranted = onPermissionGranted
        this.onPermissionDenied = onPermissionDenied
        // 检查权限状态并相应地处理。
        when {
            ContextCompat.checkSelfPermission(
                this, permission.permissionType
            ) == PackageManager.PERMISSION_GRANTED -> onPermissionGranted.invoke()
            ActivityCompat.shouldShowRequestPermissionRationale(
                this, permission.permissionType
            )                                      -> showRequestPermissionRationaleDialog(
                permission
            )
            else                                   -> {
                requestPermissionLauncher.launch(permission.permissionType)
                explainAccessToMoreSensitiveInformation(permission)
            }
        }
    }

    override fun showRequestPermissionRationaleDialog(
        permission: Permission
    ) {
        AlertDialog
            .Builder(this@BasePermissionActivity)
            .setMessage("你已禁止授予应用 ${permission.permissionName} 权限，可能会造成功能不可用，如需使用请到设置里授予权限")
            .setNegativeButton(
                "取消"
            ) { dialog, _ ->
                dialog.dismiss()
            }
            .setPositiveButton("前往授权") { dialog, _ ->
                dialog.dismiss()
                grantingPermissions()
            }
            .show()
    }

    override fun grantingPermissions() {
        val intent = Intent()
        intent.action = Settings.ACTION_APPLICATION_DETAILS_SETTINGS
        val uri = Uri.fromParts(
            "package", packageName, null
        )
        intent.data = uri
        if (intent.resolveActivity(packageManager) != null) {
            startActivity(intent)
        } else {
            Errors.show("无法打开应用信息页面")
        }
    }

    override fun getExplanationForSensitiveInformation(permission: Permission): String? {
        return when (permission) {
            Permission.CAMERA                -> "应用需要申请相机权限，以便您可以正常使用扫码，实名认证，拍摄和上传照片等服务"
            Permission.READ_PHONE_STATE      -> "为了信息推送和账号安全，我们会申请系统设备权限收集设备信息、日志信息等"
            Permission.READ_CONTACTS         -> "用用需要申请通讯录权限，用以匹配或添加联系人（未经您同意，我们不会从第三方获取、共享或对外提供您的信息）"
            Permission.READ_EXTERNAL_STORAGE -> "应用需要访问您的存储权限，以便您可以正常使用文件导入、签字、发送、下载、实名认证等服务"
            else                             -> null
        }
    }

    override fun explainAccessToMoreSensitiveInformation(
        permission: Permission
    ) {
        val explanation = getExplanationForSensitiveInformation(permission) ?: return
        permissionExplanationDialog = AlertDialog
            .Builder(this@BasePermissionActivity)
            .setTitle("权限使用说明")
            .setMessage(explanation)
            .create()
            .apply {
                setOnShowListener {
                    permissionExplanationDialog?.window?.setGravity(Gravity.TOP)
                }
            }
        permissionExplanationDialog?.show()
    }

}