package com.liecoder.framework.base

import android.content.pm.ActivityInfo.SCREEN_ORIENTATION_PORTRAIT
import android.graphics.drawable.Drawable
import android.view.Gravity
import android.widget.Toast
import com.liecoder.framework.toast.ToastStyle

/**
 * 基础服务接口，提供一组标准方法来简化活动（Activity）的初始化和状态管理。
 *
 * 实现此接口的类将负责执行与设置内容视图、初始化、视图设置、数据加载、状态栏配置、加载指示器控制、
 * 消息提示等相关的基本操作。这些方法旨在作为应用程序中所有活动的基础实现。
 *
 * @author liecoder
 * @date 2024/6/20 周四
 * @version 1.0
 */
interface IBaseService {

    /**
     * 设置内容视图的方法。
     *
     * 此方法应设置活动的布局，通常是通过调用 setContentView() 来实现。
     */
    fun setContentView()


    /**
     * 初始化方法。
     *
     *在这个方法中，应执行所有必要的初始化操作，如依赖注入或初始状态设置。
     */
    fun init()


    /**
     * 设置视图的方法。
     *
     * 此方法用于配置活动的 UI 组件，例如设置监听器或初始视图状态。
     */
    fun setupViews()


    /**
     * 加载数据的方法。
     *
     * 在这个方法中，应执行数据加载逻辑，如从网络或本地数据库获取数据。
     */
    fun loadData()

    fun getScreenOrientation():Int = SCREEN_ORIENTATION_PORTRAIT


    /**
     * 设置状态栏的方法。
     *
     * 此方法用于配置活动的状态栏，例如状态栏颜色、是否深色内容等。
     */
    fun setupStatusBar()


    /**
     * 判断状态栏内容是否为深色。
     *
     * 返回布尔值，指示状态栏的内容颜色是否为深色。
     * @return 如果状态栏内容颜色为深色，则返回 true。
     */
    fun statusBarDarkContent(): Boolean


    /**
     * 获取状态栏颜色。
     *
     * 返回整数值，表示状态栏的颜色。
     * @return 状态栏的颜色值。
     */
    fun statusBarColor(): Int


    /**
     * 判断状态栏是否半透明。
     *
     * 返回布尔值，指示状态栏是否设置为半透明。
     * @return 如果状态栏半透明，则返回 true。
     */
    fun statusBarTranslucent(): Boolean


    /**
     * 显示或隐藏Loading框。
     *
     * @param show 是否显示加载指示器。
     * @param message 可选的加载信息提示。
     */
    fun showLoading(
        show: Boolean = true, message: String? = null
    )

    /**
     * 隐藏Loading框。
     */
    fun hideLoading()


    /**
     * 显示成功吐司消息。
     *
     * @param message 要显示的成功消息。
     */
    fun success(message: CharSequence?)


    /**
     * 显示错误吐司消息。
     *
     * @param message 要显示的错误消息。
     */
    fun errors(message: CharSequence?)


    /**
     * 显示警告吐司消息。
     *
     * @param message 要显示的警告消息。
     */
    fun warnings(message: CharSequence?)


    /**
     * 显示信息吐司消息。
     *
     * @param message 要显示的信息消息。
     */
    fun info(message: CharSequence?)


    /**
     * 显示自定义吐司消息。
     *
     * @param message 要显示的消息文本。
     * @param successIcon 成功图标。
     * @param errorIcon 错误图标。
     * @param warningIcon 警告图标。
     * @param gravity 吐司显示位置。
     * @param xOffset 吐司水平偏移量。
     * @param yOffset 吐司垂直偏移量。
     * @param duration 吐司显示时长。
     * @param style 吐司样式。
     */
    fun custom(
        message: CharSequence?,
        successIcon: Drawable? = null,
        errorIcon: Drawable? = null,
        warningIcon: Drawable? = null,
        gravity: Int = Gravity.CENTER,
        xOffset: Int = 0,
        yOffset: Int = 0,
        duration: Int = Toast.LENGTH_LONG,
        style: ToastStyle? = null
    )

}