package com.liecoder.framework.ktx

import android.os.SystemClock
import android.view.View
import android.widget.CheckBox
import com.liecoder.framework.R

/**
 * 设置一个防抖的点击事件监听器。
 *
 * 此函数为任意视图 [V] 设置一个点击事件监听器，增加了时间间隔防抖的功能。如果两次点击的时间间隔小于指定的 [wait] 参数，则第二次点击不会触发事件。
 *
 * @param wait 点击间隔的防抖时间，单位为毫秒，默认为200毫秒。
 * @param onClickBlock 点击事件的回调函数，当视图被点击且满足防抖条件时执行。
 * @return 返回视图本身，允许链式调用。
 */
fun <V : View> V.onClick(
    wait: Long = 200, onClickBlock: (view: View) -> Unit
): V {
    setOnClickListener { view ->
        val currentTimeMillis = SystemClock.uptimeMillis()
        val lastClickTime = (view.getTag(R.id.click_time_stamp) as? Long) ?: 0
        if (currentTimeMillis - lastClickTime > wait) {
            view.setTag(
                R.id.click_time_stamp, currentTimeMillis
            )
            onClickBlock(view)
        }
    }
    return this
}

/**
 * 设置视图的点击启用状态。
 * 当启用时，视图将响应点击事件，并且透明度设置为完全不透明。
 * 当禁用时，视图将不响应点击事件，并且透明度会降低。
 *
 * @param enabled 是否启用点击。默认为 true。
 * @param disabledAlpha 禁用时的透明度。默认为 0.3f。
 * @return V 原视图实例，方便链式调用。
 */
fun <V : View> V.setClickEnabled(
    enabled: Boolean = true, disabledAlpha: Float = 0.3f
): V {
    isClickable = enabled
    alpha = if (enabled) 1.0F else disabledAlpha
    (this as? CheckBox)?.isEnabled = enabled
    return this
}

/**
 * * 设置视图的点击启用状态。
 * 当启用时，视图将响应点击事件，并且透明度设置为完全不透明。
 * 当禁用时，视图将不响应点击事件，并且透明度会降低。
 *
 * @param disabled 是否禁用点击。默认为 true，意味着调用此函数将禁用点击。
 * @param disabledAlpha 禁用时的透明度。默认为 0.3f。
 * @return V 原视图实例，方便链式调用。
 */
fun <V : View> V.setClickDisabled(disabled: Boolean = true, disabledAlpha: Float = 0.3f): V {
    isClickable = !disabled
    alpha = if (!disabled) 1.0F else disabledAlpha
    (this as? CheckBox)?.isEnabled = !disabled
    return this
}
