package com.liecoder.framework.ktx

import android.graphics.Paint
import android.graphics.Typeface
import android.graphics.drawable.Drawable
import android.text.TextUtils.TruncateAt
import android.view.Gravity
import android.widget.TextView
import androidx.core.view.marginEnd
import androidx.core.view.marginStart


const val ELLIPSIZE_NONE: Int = 0
const val ELLIPSIZE_START: Int = 1
const val ELLIPSIZE_MIDDLE: Int = 2
const val ELLIPSIZE_END: Int = 3
const val ELLIPSIZE_MARQUEE: Int = 4


/**
 * 判断 TextView 是否包含文本内容或复合绘制图标。
 *
 * 此属性检查 TextView 的 text 属性是否有文本，如果没有文本，则进一步检查是否有设置复合绘制图标（即 Drawable）。
 * 如果 TextView 包含文本或至少有一个复合绘制图标，则认为它包含内容。
 *
 * @return 如果 TextView 包含文本或复合绘制图标，则返回 true，否则返回 false。
 */
inline val TextView.containsContent: Boolean
    get() {
        if (!text.isNullOrEmpty()) return true
        return compoundDrawables.any { it != null }
    }

/**
 * 将整数常量转换为对应的 [Typeface] 样式。
 *
 * 此属性扩展根据传入的整数常量，返回一个对应的 [Typeface] 对象。
 *
 * @return 返回对应的 [Typeface] 对象。
 */
inline val Int.asTypeface: Typeface
    get() = when (this) {
        Typeface.BOLD -> Typeface.DEFAULT_BOLD
        Typeface.ITALIC, Typeface.BOLD_ITALIC -> Typeface.MONOSPACE
        else -> Typeface.DEFAULT
    }

/**
 * 将整数常量转换为对应的 [TruncateAt] 枚举值。
 *
 * 此属性扩展根据传入的整数常量，返回一个对应的 [TruncateAt] 枚举值，用于指定文本的省略显示方式。
 *
 * @return 返回对应的 [TruncateAt] 枚举值，如果无法匹配则返回 null。
 */
inline val Int.asTruncateAt: TruncateAt?
    get() = when (this) {
        ELLIPSIZE_START -> TruncateAt.START
        ELLIPSIZE_MIDDLE -> TruncateAt.MIDDLE
        ELLIPSIZE_END -> TruncateAt.END
        ELLIPSIZE_MARQUEE -> TruncateAt.MARQUEE
        else -> null
    }

/**
 * 应用文本省略效果到 TextView。
 *
 * 此函数根据传入的 [TruncateAt] 值设置 TextView 的省略效果。如果设置为 [TruncateAt.MARQUEE]，则还会启用滚动效果，并确保 TextView 是可选的和可聚焦的。
 *
 * @param where 指定文本省略的位置，可以是 [TruncateAt.START]、[TruncateAt.MIDDLE]、[TruncateAt.END] 或 [TruncateAt.MARQUEE]。
 */
fun TextView.applyEllipsize(where: TruncateAt?) {
    if (ellipsize == where) return
    ellipsize = where
    if (where != TruncateAt.MARQUEE) return
    if (!isSelected) isSelected = true
    if (!isFocusable) isFocusable = true
    marqueeRepeatLimit = -1
}

/**
 * 根据指定的方向获取对应的复合绘制图标（Compound Drawable）。
 *
 * 此函数通过传入的 [gravity] 参数，使用 [getAbsoluteGravity] 方法转换为绝对方向，并根据该值从 TextView 的复合绘制图标中返回对应的图标。
 * 支持的方向包括左（[Gravity.LEFT]）、顶部（[Gravity.TOP]）、右（[Gravity.RIGHT]）和底部（[Gravity.BOTTOM]）。
 *
 * @param gravity 指定的方向，用于确定从哪个位置的复合绘制图标获取图标。
 * @return 返回对应方向的 Drawable 对象，如果没有找到对应的图标则返回 null。
 */
fun TextView.getCompoundDrawableGravity(gravity: Int): Drawable? =
    when (getAbsoluteGravity(gravity)) {
        Gravity.LEFT -> compoundDrawables[0]
        Gravity.TOP -> compoundDrawables[1]
        Gravity.RIGHT -> compoundDrawables[2]
        Gravity.BOTTOM -> compoundDrawables[3]
        else -> null
    }

/**
 * 根据指定的方向设置 TextView 的复合绘制图标（Compound Drawable）。
 *
 * 此函数接受一个方向 [gravity] 和一个 [Drawable] 对象，然后根据方向确定 Drawable 应该放置在 TextView 的哪个位置。
 * 支持的方向包括 [Gravity.LEFT]、[Gravity.TOP]、[Gravity.RIGHT] 和 [Gravity.BOTTOM]。
 *
 * @param gravity 指定的方向，用于确定 Drawable 放置的位置。
 * @param drawable 要设置的 Drawable 对象，可以为 null。
 * @return 返回 TextView 实例自身，允许链式调用。
 */
fun TextView.setCompoundDrawableGravity(gravity: Int, drawable: Drawable?): TextView {
    when (getAbsoluteGravity(gravity)) {
        Gravity.LEFT -> setCompoundDrawables(drawable, null, null, null)
        Gravity.TOP -> setCompoundDrawables(null, drawable, null, null)
        Gravity.RIGHT -> setCompoundDrawables(null, null, drawable, null)
        Gravity.BOTTOM -> setCompoundDrawables(null, null, null, drawable)
        else -> setCompoundDrawables(null, null, null, null)
    }
    return this
}

/**
 * 计算并返回 TextView 的总宽度。
 *
 * 此函数通过测量文本的宽度并加上内边距（padding）、外边距（margin）来计算 TextView 的总宽度。
 * 这包括文本自身所占用的宽度以及 TextView 的内边距和外边距。
 *
 * @return TextView 的总宽度，以像素为单位的浮点数。
 */
fun TextView.getFullWidth(): Float {
    return paint.measureText(text.toString()) + paddingStart + paddingEnd + marginStart + marginEnd
}

/**
 * 获取 TextView 的文本是否启用了删除线（中划线）样式。
 *
 * 此属性通过检查 TextView 的 paintFlags 与删除线的 flag 进行与操作来确定是否启用了删除线样式。
 *
 * @return 如果启用了删除线样式返回 true，否则返回 false。
 */
inline val TextView.isStrikethroughEnabled: Boolean
    get() = paintFlags and Paint.STRIKE_THRU_TEXT_FLAG != 0

/**
 * 设置 TextView 的文本是否显示为删除线（中划线）样式。
 *
 * 此函数根据参数 [enabled] 的值来启用或禁用删除线样式。当 [enabled] 为 true 时，文本将显示删除线；为 false 时，删除线将被移除。
 * 默认情况下，[enabled] 参数为 true，表示调用此函数时若不指定参数，将启用删除线样式。
 *
 * @param enabled 一个布尔值，指示是否启用删除线样式。
 * @return 返回 TextView 实例自身，允许链式调用。
 */
fun TextView.setStrikethrough(enabled: Boolean = true): TextView {
    paintFlags = if (enabled) {
        paintFlags or Paint.STRIKE_THRU_TEXT_FLAG
    } else {
        paintFlags and Paint.STRIKE_THRU_TEXT_FLAG.inv()
    }
    return this
}

/**
 * 使 TextView 的文本显示为粗体样式。
 *
 * 此函数通过设置 paint 的 isFakeBoldText 属性为 true 来模拟粗体效果，不改变原始字体的样式。
 *
 * @return 返回 TextView 实例自身，允许链式调用。
 */
fun TextView.bold(): TextView {
    paint.isFakeBoldText = true // 设置为 true 以模拟粗体效果
    return this
}

/**
 * 应用文本样式到 TextView。
 *
 * 此函数通过设置 [Typeface] 来改变 TextView 的字体样式。如果当前 TextView 没有设置 typeface，则使用系统默认的 typeface。
 * 可以指定一个样式参数，如 [Typeface.BOLD] 或 [Typeface.ITALIC]，来应用不同的文本样式。
 *
 * @param style 字体样式，可以是 [Typeface.BOLD]、[Typeface.ITALIC] 或它们的组合。
 * @return 返回 TextView 实例自身，允许链式调用。
 */
fun TextView.applyTextStyle(style: Int = Typeface.BOLD): TextView {
    typeface = typeface ?: Typeface.defaultFromStyle(style)
    return this
}

/**
 * 设置 TextView 的相对位置复合绘制图标。
 *
 * 此函数为 TextView 设置相对于文本内容的左侧、顶部、右侧和底部的图标，使用 [setCompoundDrawablesRelativeWithIntrinsicBounds] 方法。
 * 这允许图标尺寸与文本行高度保持一致，同时支持 RTL（从右到左）布局。
 *
 * @param start 绘制在文本开始侧的图标，对于 LTR（从左到右）布局是左侧，对于 RTL 布局是右侧。
 * @param top 绘制在文本顶部的图标。
 * @param end 绘制在文本结束侧的图标，对于 LTR 布局是右侧，对于 RTL 布局是左侧。
 * @param bottom 绘制在文本底部的图标。
 * @return 返回 TextView 实例自身，允许链式调用。
 */
fun TextView.compoundDrawable(
    start: Drawable? = null,
    top: Drawable? = null,
    end: Drawable? = null,
    bottom: Drawable? = null
): TextView {
    setCompoundDrawablesRelativeWithIntrinsicBounds(start, top, end, bottom)
    return this
}
