package com.liecoder.framework.utils

import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import java.io.FileOutputStream
import java.net.URL

object HttpDownloadUtils {

    /**
     * 从指定的 URL 下载文件并保存到本地路径。
     *
     * @param fileUrl 文件的 URL 地址。
     * @param destinationPath 文件保存的本地路径。
     */
    suspend fun downloadFile(
        fileUrl: String, destinationPath: String
    ): Result<Unit> = withContext(Dispatchers.IO) {
        try {
            // 创建 URL 对象
            val url = URL(fileUrl)
            // 打开连接
            val connection = url.openConnection()
            // 使用 use 函数自动关闭流
            connection.getInputStream().use { inputStream ->
                FileOutputStream(destinationPath).use { outputStream ->
                    // 创建缓冲区
                    val buffer = ByteArray(1024)
                    var bytesRead: Int
                    // 读取数据并写入文件
                    while (inputStream.read(buffer).also { bytesRead = it } != -1) {
                        outputStream.write(buffer, 0, bytesRead)
                    }
                }
            }
            Result.success(Unit)
        } catch (e: Exception) {
            // 打印异常信息
            e.printStackTrace()
            Result.failure(Exception("Failed to download file: ${e.message}"))
        }
    }

}
