/*
 * Decompiled with CFR 0.152.
 */
package io.microsphere.reflect;

import io.microsphere.annotation.Nonnull;
import io.microsphere.annotation.Nullable;
import io.microsphere.collection.ListUtils;
import io.microsphere.lang.function.Predicates;
import io.microsphere.lang.function.Streams;
import io.microsphere.logging.Logger;
import io.microsphere.logging.LoggerFactory;
import io.microsphere.reflect.AccessibleObjectUtils;
import io.microsphere.reflect.MemberUtils;
import io.microsphere.reflect.TypeUtils;
import io.microsphere.text.FormatUtils;
import io.microsphere.util.AnnotationUtils;
import io.microsphere.util.ArrayUtils;
import io.microsphere.util.ClassUtils;
import io.microsphere.util.Utils;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.function.Predicate;

public abstract class MethodUtils
implements Utils {
    private static final Logger logger = LoggerFactory.getLogger(MethodUtils.class);
    public static final List<Method> OBJECT_PUBLIC_METHODS = ListUtils.of(Object.class.getMethods());
    public static final List<Method> OBJECT_DECLARED_METHODS = ListUtils.of(Object.class.getDeclaredMethods());
    public static final Predicate<? super Method> OBJECT_METHOD_PREDICATE = MethodUtils::isObjectMethod;
    public static final Predicate<? super Method> PUBLIC_METHOD_PREDICATE = MemberUtils::isPublic;
    public static final Predicate<? super Method> STATIC_METHOD_PREDICATE = MemberUtils::isStatic;
    public static final Predicate<? super Method> NON_STATIC_METHOD_PREDICATE = MemberUtils::isNonStatic;
    public static final Predicate<? super Method> FINAL_METHOD_PREDICATE = MemberUtils::isFinal;
    public static final Predicate<? super Method> NON_PRIVATE_METHOD_PREDICATE = MemberUtils::isNonPrivate;
    private static final ConcurrentMap<MethodKey, Method> methodsCache = new ConcurrentHashMap<MethodKey, Method>(256);
    private static final ConcurrentMap<Class<?>, Method[]> declaredMethodsCache = new ConcurrentHashMap(256);

    @Nonnull
    public static Predicate<? super Method> excludedDeclaredClass(Class<?> declaredClass) {
        return method -> !Objects.equals(declaredClass, method.getDeclaringClass());
    }

    @Nonnull
    public static List<Method> getDeclaredMethods(Class<?> targetClass) {
        return MethodUtils.findDeclaredMethods(targetClass, Predicates.EMPTY_PREDICATE_ARRAY);
    }

    @Nonnull
    public static List<Method> getMethods(Class<?> targetClass) {
        return MethodUtils.findMethods(targetClass, Predicates.EMPTY_PREDICATE_ARRAY);
    }

    @Nonnull
    public static List<Method> getAllDeclaredMethods(Class<?> targetClass) {
        return MethodUtils.findAllDeclaredMethods(targetClass, Predicates.EMPTY_PREDICATE_ARRAY);
    }

    @Nonnull
    public static List<Method> getAllMethods(Class<?> targetClass) {
        return MethodUtils.findAllMethods(targetClass, Predicates.EMPTY_PREDICATE_ARRAY);
    }

    @Nonnull
    public static List<Method> findDeclaredMethods(Class<?> targetClass, Predicate<? super Method> ... methodsToFilter) {
        return MethodUtils.findMethods(targetClass, false, false, methodsToFilter);
    }

    @Nonnull
    public static List<Method> findMethods(Class<?> targetClass, Predicate<? super Method> ... methodsToFilter) {
        return MethodUtils.findMethods(targetClass, false, true, methodsToFilter);
    }

    @Nonnull
    public static List<Method> findAllDeclaredMethods(Class<?> targetClass, Predicate<? super Method> ... methodsToFilter) {
        return MethodUtils.findMethods(targetClass, true, false, methodsToFilter);
    }

    @Nonnull
    public static List<Method> findAllMethods(Class<?> targetClass, Predicate<? super Method> ... methodsToFilter) {
        return MethodUtils.findMethods(targetClass, true, true, methodsToFilter);
    }

    @Nonnull
    public static List<Method> findMethods(Class<?> targetClass, boolean includeInheritedTypes, boolean publicOnly, Predicate<? super Method> ... methodsToFilter) {
        if (targetClass == null || ClassUtils.isPrimitive(targetClass)) {
            return Collections.emptyList();
        }
        if (ClassUtils.isArray(targetClass)) {
            return MethodUtils.doFilterMethods(OBJECT_PUBLIC_METHODS, methodsToFilter);
        }
        if (TypeUtils.isObjectClass(targetClass)) {
            return publicOnly ? MethodUtils.doFilterMethods(OBJECT_PUBLIC_METHODS, methodsToFilter) : MethodUtils.doFilterMethods(OBJECT_DECLARED_METHODS, methodsToFilter);
        }
        Predicate<? super Method> predicate = Predicates.and(methodsToFilter);
        if (publicOnly) {
            predicate = PUBLIC_METHOD_PREDICATE.and(predicate);
        }
        LinkedList<Method> allMethods = new LinkedList<Method>();
        if (includeInheritedTypes) {
            while (targetClass != null) {
                MethodUtils.filterDeclaredMethodsHierarchically(targetClass, predicate, allMethods);
                targetClass = targetClass.getSuperclass();
            }
        } else {
            MethodUtils.filterDeclaredMethods(targetClass, predicate, allMethods);
        }
        return Collections.unmodifiableList(allMethods);
    }

    @Nullable
    public static Method findMethod(Class targetClass, String methodName) {
        return MethodUtils.findMethod(targetClass, methodName, ArrayUtils.EMPTY_CLASS_ARRAY);
    }

    @Nullable
    public static Method findMethod(Class targetClass, String methodName, Class<?> ... parameterTypes) {
        MethodKey key = MethodUtils.buildKey(targetClass, methodName, parameterTypes);
        return methodsCache.computeIfAbsent(key, MethodUtils::doFindMethod);
    }

    @Nullable
    public static Method findDeclaredMethod(Class<?> targetClass, String methodName, Class<?> ... parameterTypes) {
        if (targetClass == null) {
            return null;
        }
        Method method = MethodUtils.doFindDeclaredMethod(targetClass, methodName, parameterTypes);
        if (method == null) {
            Class superClass = targetClass.isInterface() ? Object.class : targetClass.getSuperclass();
            method = MethodUtils.findDeclaredMethod(superClass, methodName, parameterTypes);
        }
        if (method == null) {
            Class<?> interfaceClass;
            Class<?>[] classArray = targetClass.getInterfaces();
            int n = classArray.length;
            for (int i = 0; i < n && (method = MethodUtils.findDeclaredMethod(interfaceClass = classArray[i], methodName, parameterTypes)) == null; ++i) {
            }
        }
        if (method == null && logger.isTraceEnabled()) {
            logger.trace("The declared method was not found in the target class[name : '{}'] by name['{}'] and parameter types['{}']", targetClass, methodName, ArrayUtils.arrayToString(parameterTypes));
        }
        return method;
    }

    @Nullable
    public static <R> R invokeMethod(Object object, String methodName, Object ... arguments) {
        Class<?> type = object.getClass();
        return MethodUtils.invokeMethod(object, type, methodName, arguments);
    }

    @Nullable
    public static <R> R invokeStaticMethod(Class<?> targetClass, String methodName, Object ... arguments) {
        return MethodUtils.invokeMethod(null, targetClass, methodName, arguments);
    }

    @Nullable
    public static <R> R invokeStaticMethod(Method method, Object ... arguments) {
        return MethodUtils.invokeMethod(null, method, arguments);
    }

    @Nullable
    public static <R> R invokeMethod(Object instance, Class<?> type, String methodName, Object ... arguments) {
        Class[] parameterTypes = ClassUtils.getTypes(arguments);
        Method method = MethodUtils.findMethod(type, methodName, parameterTypes);
        if (method == null) {
            throw new IllegalStateException(FormatUtils.format("cannot find method[name : '{}'], class: '{}'", methodName, type.getName()));
        }
        return MethodUtils.invokeMethod(instance, method, arguments);
    }

    @Nullable
    public static <R> R invokeMethod(@Nullable Object instance, Method method, Object ... arguments) {
        if (method == null) {
            throw new NullPointerException("The 'method' must not be null");
        }
        Object result = null;
        boolean accessible = false;
        Throwable failure = null;
        try {
            accessible = AccessibleObjectUtils.trySetAccessible(method);
            result = method.invoke(instance, arguments);
        }
        catch (IllegalAccessException e) {
            String errorMessage = FormatUtils.format("The method[signature : '{}' , instance : {}] can't be accessed[accessible : {}]", MethodUtils.getSignature(method), instance, accessible);
            failure = new IllegalStateException(errorMessage, e);
        }
        catch (IllegalArgumentException e) {
            String errorMessage = FormatUtils.format("The arguments can't match the method[signature : '{}' , instance : {}] : {}", MethodUtils.getSignature(method), instance, ArrayUtils.arrayToString(arguments));
            failure = new IllegalArgumentException(errorMessage, e);
        }
        catch (InvocationTargetException e) {
            String errorMessage = FormatUtils.format("It's failed to invoke the method[signature : '{}' , instance : {} , arguments : {}]", MethodUtils.getSignature(method), instance, ArrayUtils.arrayToString(arguments));
            failure = new RuntimeException(errorMessage, e.getTargetException());
        }
        if (failure != null) {
            logger.error(failure.getMessage(), failure.getCause());
            throw failure;
        }
        return (R)result;
    }

    public static boolean overrides(Method overrider, Method overridden) {
        Class<?>[] overriddenParameterTypes;
        Class<?> overriddenDeclaringClass;
        if (overrider == null || overridden == null || overrider == overridden) {
            return false;
        }
        if (!Objects.equals(overrider.getName(), overridden.getName())) {
            return false;
        }
        if (MemberUtils.isStatic(overrider) || MemberUtils.isStatic(overridden)) {
            return false;
        }
        if (MemberUtils.isPrivate(overrider) || MemberUtils.isPrivate(overridden)) {
            return false;
        }
        if (overrider.isDefault()) {
            return false;
        }
        Class<?> overriderDeclaringClass = overrider.getDeclaringClass();
        if (overriderDeclaringClass == (overriddenDeclaringClass = overridden.getDeclaringClass())) {
            return false;
        }
        if (!overriddenDeclaringClass.isAssignableFrom(overriderDeclaringClass)) {
            return false;
        }
        int parameterCount = overrider.getParameterCount();
        if (parameterCount != overridden.getParameterCount()) {
            return false;
        }
        Class<?>[] overriderParameterTypes = overrider.getParameterTypes();
        if (!MethodUtils.matchesParameterTypes(overriderParameterTypes, overriddenParameterTypes = overridden.getParameterTypes(), parameterCount)) {
            return false;
        }
        return overridden.getReturnType().isAssignableFrom(overrider.getReturnType());
    }

    @Nullable
    public static Method findNearestOverriddenMethod(Method overrider) {
        Class<?> inheritedType;
        Class<?> targetClass = overrider.getDeclaringClass();
        Method overriddenMethod = null;
        Iterator<Class<?>> iterator = ClassUtils.getAllInheritedTypes(targetClass).iterator();
        while (iterator.hasNext() && (overriddenMethod = MethodUtils.findOverriddenMethod(overrider, inheritedType = iterator.next())) == null) {
        }
        return overriddenMethod;
    }

    @Nullable
    public static Method findOverriddenMethod(Method overrider, Class<?> targetClass) {
        List<Method> matchedMethods = MethodUtils.findDeclaredMethods(targetClass, method -> MethodUtils.overrides(overrider, method));
        return matchedMethods.isEmpty() ? null : matchedMethods.get(0);
    }

    @Nonnull
    public static String getSignature(Method method) {
        return MethodUtils.buildSignature(method.getDeclaringClass(), method.getName(), method.getParameterTypes());
    }

    static String buildSignature(Class<?> declaringClass, String methodName, Class<?>[] parameterTypes) {
        String parameterTypeName;
        int parameterCount = parameterTypes.length;
        String[] parameterTypeNames = new String[parameterCount];
        String declaringClassName = ClassUtils.getTypeName(declaringClass);
        int size = declaringClassName.length() + 1 + methodName.length() + 1 + (parameterCount == 0 ? 0 : parameterCount - 1) + 1;
        for (int i = 0; i < parameterCount; ++i) {
            Class<?> parameterType = parameterTypes[i];
            parameterTypeNames[i] = parameterTypeName = ClassUtils.getTypeName(parameterType);
            size += parameterTypeName.length();
        }
        StringBuilder signatureBuilder = new StringBuilder(size);
        signatureBuilder.append(declaringClassName).append('#').append(methodName).append('(');
        for (int i = 0; i < parameterCount; ++i) {
            parameterTypeName = parameterTypeNames[i];
            signatureBuilder.append(parameterTypeName);
            if (i < parameterCount - 1) {
                signatureBuilder.append(',');
            }
            parameterTypeNames[i] = null;
        }
        signatureBuilder.append(')');
        return signatureBuilder.toString();
    }

    public static boolean isObjectMethod(Method method) {
        if (method != null) {
            return TypeUtils.isObjectClass(method.getDeclaringClass());
        }
        return false;
    }

    public static boolean isCallerSensitiveMethod(Method method) {
        return AnnotationUtils.isAnnotationPresent((AnnotatedElement)method, AnnotationUtils.CALLER_SENSITIVE_ANNOTATION_CLASS);
    }

    static void filterDeclaredMethodsHierarchically(Class<?> targetClass, Predicate<? super Method> methodToFilter, List<Method> methodsToCollect) {
        MethodUtils.filterDeclaredMethods(targetClass, methodToFilter, methodsToCollect);
        for (Class<?> interfaceClass : targetClass.getInterfaces()) {
            MethodUtils.filterDeclaredMethodsHierarchically(interfaceClass, methodToFilter, methodsToCollect);
        }
    }

    static void filterDeclaredMethods(Class<?> targetClass, Predicate<? super Method> methodToFilter, List<Method> methodsToCollect) {
        for (Method method : MethodUtils.doGetDeclaredMethods(targetClass)) {
            if (!methodToFilter.test(method)) continue;
            methodsToCollect.add(method);
        }
    }

    static Method doFindDeclaredMethod(Class<?> klass, String methodName, Class<?>[] parameterTypes) {
        Method[] declaredMethods = MethodUtils.doGetDeclaredMethods(klass);
        return MethodUtils.doFindMethod(declaredMethods, methodName, parameterTypes);
    }

    static Method doFindMethod(Method[] methods, String methodName, Class<?>[] parameterTypes) {
        Method targetMethod = null;
        for (Method method : methods) {
            if (!MethodUtils.matches(method, methodName, parameterTypes)) continue;
            targetMethod = method;
            break;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("The target method[name : '{}' , parameter types : {}] {} found in the methods : {}", methodName, ArrayUtils.arrayToString(parameterTypes), targetMethod == null ? "can't be" : "is", ArrayUtils.arrayToString(methods));
        }
        return targetMethod;
    }

    static boolean matches(Method method, String methodName, Class<?>[] parameterTypes) {
        int parameterCount = parameterTypes.length;
        return parameterCount == method.getParameterCount() && Objects.equals(method.getName(), methodName) && MethodUtils.matchesParameterTypes(method.getParameterTypes(), parameterTypes, parameterCount);
    }

    static boolean matchesParameterTypes(Class<?>[] oneParameterTypes, Class<?>[] anotherParameterTypes, int parameterCount) {
        for (int i = 0; i < parameterCount; ++i) {
            if (oneParameterTypes[i] == anotherParameterTypes[i]) continue;
            return false;
        }
        return true;
    }

    static Method[] doGetDeclaredMethods(Class<?> klass) {
        return declaredMethodsCache.computeIfAbsent(klass, c -> c.getDeclaredMethods());
    }

    static List<Method> doFilterMethods(List<Method> methods, Predicate<? super Method> ... methodsToFilter) {
        return Collections.unmodifiableList(Streams.filterAll(methods, methodsToFilter));
    }

    static MethodKey buildKey(Class<?> declaredClass, String methodName, Class<?> ... parameterTypes) {
        return new MethodKey(declaredClass, methodName, parameterTypes);
    }

    static Method doFindMethod(MethodKey key) {
        Class<?> declaredClass = key.declaredClass;
        String methodName = key.methodName;
        Class<?>[] parameterTypes = key.parameterTypes;
        return MethodUtils.findDeclaredMethod(declaredClass, methodName, parameterTypes);
    }

    private MethodUtils() {
    }

    static class MethodKey {
        final Class<?> declaredClass;
        final String methodName;
        final Class<?>[] parameterTypes;

        MethodKey(Class<?> declaredClass, String methodName, Class<?>[] parameterTypes) {
            this.declaredClass = declaredClass;
            this.methodName = methodName;
            this.parameterTypes = parameterTypes == null ? ArrayUtils.EMPTY_CLASS_ARRAY : parameterTypes;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            MethodKey methodKey = (MethodKey)o;
            if (!Objects.equals(this.declaredClass, methodKey.declaredClass)) {
                return false;
            }
            if (!Objects.equals(this.methodName, methodKey.methodName)) {
                return false;
            }
            return ArrayUtils.arrayEquals(this.parameterTypes, methodKey.parameterTypes);
        }

        public int hashCode() {
            int result = this.declaredClass != null ? this.declaredClass.hashCode() : 0;
            result = 31 * result + (this.methodName != null ? this.methodName.hashCode() : 0);
            result = 31 * result + Objects.hash(this.parameterTypes);
            return result;
        }

        public String toString() {
            return MethodUtils.buildSignature(this.declaredClass, this.methodName, this.parameterTypes);
        }
    }
}

