package io.github.opensabe.mapstruct.processor;

import freemarker.template.Configuration;
import freemarker.template.Template;
import freemarker.template.TemplateException;
import freemarker.template.Version;
import io.github.opensabe.mapstruct.core.CommonCopyMapper;
import io.github.opensabe.mapstruct.core.CustomerMapper;
import io.github.opensabe.mapstruct.core.FromMapMapper;
import org.mapstruct.Mapper;

import javax.annotation.processing.*;
import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.Element;
import javax.lang.model.element.PackageElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.Elements;
import javax.lang.model.util.Types;
import java.io.IOException;
import java.io.Writer;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

/**
 * 将{@link io.github.opensabe.mapstruct.core.RegisterMapper} 的接口注册到{@link io.github.opensabe.mapstruct.core.MapperRepository}
 * @author heng.ma
 */
@SupportedAnnotationTypes("io.github.opensabe.mapstruct.core.RegisterMapper")
public class MapperRegisterProcessor extends AbstractProcessor {

    private Types typeUtils;
    private Elements elementUtils;
    private Element commonMapper;
    private Element mapMapper;
    private Filer filer;
    private Set<MapperRegister> mappers = new HashSet<>();
    private TypeMirror mapperMirror;
    @Override
    public synchronized void init(ProcessingEnvironment processingEnv) {
        super.init(processingEnv);
        typeUtils = processingEnv.getTypeUtils();
        elementUtils = processingEnv.getElementUtils();
        commonMapper = elementUtils.getTypeElement(CommonCopyMapper.class.getName());
        mapMapper = elementUtils.getTypeElement(FromMapMapper.class.getName());
        filer = processingEnv.getFiler();
        mapperMirror = elementUtils.getTypeElement(Mapper.class.getName()).asType();
    }

    @Override
    public boolean process(Set<? extends TypeElement> annotations, RoundEnvironment roundEnv) {
        if (!roundEnv.processingOver()) {
            for (TypeElement annotation : annotations) {
                Set<? extends Element> mappers = roundEnv.getElementsAnnotatedWith(annotation);
                for (Element mapper : mappers) {
                    if (hasMapperAnnotation(mapper) && mapper instanceof TypeElement typeElement) {
                        typeElement.getInterfaces().stream().filter(this::resolveType)
                                .forEach(typeMirror -> {
                                    if (typeMirror instanceof DeclaredType declaredType) {
                                        var args = declaredType.getTypeArguments();
                                        if (args.size() == 2) {
                                            this.mappers.add(new MapperRegister(mapper.toString(), args.get(0).toString(), args.get(1).toString()));
                                        }else {
                                            this.mappers.add(new MapperRegister(mapper.toString(), null, args.get(0).toString()));
                                        }
                                    }
                                });
                    }
                }
            }
        }else {
            try (Writer writer = filer.createSourceFile(CustomerMapper.class.getName()+"Impl")
                    .openWriter()) {
                Configuration cfg = new Configuration(new Version("2.3.32"));
                cfg.setClassForTemplateLoading(MapperRegisterProcessor.class, "/");
                cfg.setDefaultEncoding("UTF-8");

                Template template = cfg.getTemplate(CustomerMapper.class.getSimpleName()+".ftl");
                template.process(Map.of("mappers", this.mappers), writer);
                writer.flush();
            } catch (TemplateException | IOException ex) {
                throw new IllegalStateException(ex);
            }finally {
                mappers.clear();
            }
        }
        return false;
    }

    private boolean resolveType (TypeMirror type) {
        Element element = typeUtils.asElement(type);
        PackageElement pack = elementUtils.getPackageOf(element);
        return (pack.equals(elementUtils.getPackageOf(commonMapper)) && element.getSimpleName().equals(commonMapper.getSimpleName()))
                || (pack.equals(elementUtils.getPackageOf(mapMapper)) && element.getSimpleName().equals(mapMapper.getSimpleName()));
    }

    private boolean hasMapperAnnotation (Element mapper) {
        return mapper.getAnnotationMirrors().stream().map(AnnotationMirror::getAnnotationType)
                .anyMatch(a -> typeUtils.isSameType(a, mapperMirror));
    }

}
