package io.github.pigaut.sql;

import io.github.pigaut.sql.database.statement.*;

import javax.sql.*;

public interface Database {

    boolean isConnected();

    void openConnection();

    void closeConnection();

    /**
     * Retrieves the DataSource associated with this database.
     *
     * @return The DataSource instance used for database connections.
     */
    DataSource getDataSource();

    void createTableIfNotExists(String name, String... columns);

    void dropTable(String table);

    void renameTable(String table, String newName);

    void addColumn(String table, String columnName);

    void renameColumn(String table, String column, String newName);

    void clearTable(String table);

    DatabaseStatement insert(String table, String... columns);

    DatabaseStatement merge(String table, String primaryKey, String... columns);

    DatabaseStatement selectAll(String table);

    /**
     * Creates a statement for this database with the given sql
     *
     * @param sql the sql statement
     * @return a new DatabaseStatement
     */
    DatabaseStatement createStatement(String sql);

    /**
     * Executes a SQL statement against the database.
     *
     * @param sql The SQL statement to execute.
     */
    void execute(String sql);

    /**
     * Executes an SQL statement that may modify the database (such as INSERT, UPDATE, DELETE).
     *
     * @param sql The SQL statement to execute.
     * @return The number of rows affected by the statement.
     */
    int executeUpdate(String sql);

    /**
     * Executes a large update SQL statement that may modify the database.
     *
     * @param sql The SQL statement to execute.
     * @return The number of rows affected by the statement.
     */
    long executeLargeUpdate(String sql);

    /**
     * Executes an SQL query and processes the results using a QueryReader.
     *
     * @param sql    The SQL query to execute.
     * @param reader The QueryReader implementation to process the query results.
     */
    void executeQuery(String sql, QueryReader reader);

    /**
     * Executes an SQL query and processes the results using a QueryReader.
     *
     * @param sql    The SQL query to execute.
     * @param reader The QueryReader implementation to process the query results.
     */
    void executeRowQuery(String sql, QueryReader reader);

}
